%dw 2.7

import * from dw::core::Arrays

import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Annotations
import * from com::mulesoft::connectivity::decorator::Operation
import * from com::mulesoft::connectivity::decorator::PaginationStrategies
import * from com::mulesoft::connectivity::decorator::Trigger
import * from com::mulesoft::connectivity::Metadata
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::Types

// Type
type AccountUpdate = {name: String, email: String, createdAt?: LWDateTime, price?: Double, priority?: 1 | 2 | 3 | 4 | 5, role?: "user" | "admin"}
type Account = AccountUpdate & { id: String}

// Connection

@OperationElement
var isAlive: Operation<{}, HttpResponse, ResultFailure<HttpResponse, Error>, HttpConnection> = {
    name: "isAlive",
    displayName: "Is alive",
    executor: (parameter, connection) -> success(
        	{
                contentType: "application/json",
                status: 200,
                headers: {},
                cookies: {}
    		})
}

@TestConnectionElement()
var testConnection = {
	validate: defineTestConnection(
        isAlive,
        (response) -> { isValid: response.value.status == 200 }
    )
}

@ConnectionElement()
var basicAuthConnectionProvider = defineBasicHttpConnectionProvider<{baseUri: @DefaultValue(value = "http://localhost") String, user: String, pass: String}>(
    (parameter) -> {username: parameter.user, password: parameter.pass},
    (parameter) -> {baseUri: parameter.baseUri},
    (parameter) -> [{in: "header", name:"Customauthheader", value: "myValue,user:" ++ parameter.user}]
)

@ConnectionElement()
var apiKeyAuthConnectionProvider = defineApiKeyHttpConnectionProvider<{baseUri: String, apiKey: String}>(
    (parameter) -> {apiKey: parameter.apiKey},
    (parameter) -> {baseUri: parameter.baseUri},
    {in: 'header', name: "X-API-KEY"}
)

@ConnectionElement()
var oauth2AuthCodeConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'authorizationCode',
        authorizationUrl: "http://localhost/authorize",
        tokenUrl: "http://localhost/token",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

@ConnectionElement()
var oauth2ClientCredentialsConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'clientCredentials',
        tokenUrl: "http://localhost/token",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

@ConnectionElement()
var oauth2ImplicitConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'implicit',
        authorizationUrl: "http://localhost/authorize",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

@ConnectionElement()
var oauth2PasswordConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'password',
        tokenUrl: "http://localhost/token",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

type Error404 = Error<"404", "NOT_FOUND">

var error404 : Error404 = {
    kind: "404",
    categories : ["NOT_FOUND"]
}

type Error400 = Error<"400", "BAD_REQUEST">

var error400 : Error400 = {
    kind: "400",
    categories : ["BAD_REQUEST"]
}

// Operation
@OperationElement()
var updateAccountById : Operation<{accountId: String, account: AccountUpdate}, HttpResponse<{id: String, name: String, email: String}>, ResultFailure<HttpResponse, Error400 | Error404>, HttpConnection> = {
    name: "updateAccountById",
    displayName: "Update Account by Id",
    executor: (parameter, connection) -> parameter.accountId match {
        case "999" -> do {
            var response = {
                contentType: "application/json",
                status: 404,
                statusText: "Not Found",
                headers: {},
                cookies: {},
                body: {}
            } <~ {
                mimeType: "application/json; charset=utf-8",
                raw: "{}" as Binary,
                transportAttributes: {
                    kind: "http",
                    statusCode: 404,
                    headers: {},
                    cookies: {}
                }
            }
            ---
            failure(response, error404,response.body.detail)
        }
        case "-1" -> do {
                var response = {
                                contentType: "application/json",
                                status: 400,
                                headers: {},
                                cookies: {},
                                body: {
                                    error: "invalidAccountId",
                                    detail: "accountId cannot be a negative value"
                                }
                        }
                ---
                failure(response, error400,response.body.detail)
            }
        else ->
            success({
                contentType: "application/json",
                status: 200,
                headers: {},
                cookies: {},
                body: {
                    id: parameter.accountId,
                    name: parameter.account.name,
                    email: parameter.account.email
                }
            } <~ {
                raw: '{"id":"$(parameter.accountId)","name":"$(parameter.account.name)","email":"$(parameter.account.email)"}' as Binary,
                mimeType: "application/json; charset=utf-8",
                transportAttributes: {
                    kind: "http",
                    statusCode: 200,
                    headers: {},
                    cookies: {}
                }
            })
        }
}

@OperationElement()
var getAccounts : Operation<{offset: Number, limit: Number}, HttpResponse<{items: Array<{id: String, name: String, email: String}>}>, ResultFailure<HttpResponse, Error>, HttpConnection> = {
    name: "getAccounts",
    displayName: "Get Accounts",
    executor: (parameter, connection) -> parameter.offset match {
            case "-1" -> do {
                    var response = {
                                    contentType: "application/json",
                                    status: 400,
                                    statusText: "Bad Request",
                                    headers: {},
                                    cookies: {},
                                    body: {}
                            } <~ { raw: "{}" as Binary, mimeType: "application/json", transportAttributes: {kind: "http", statusCode: 200, headers: {}, cookies: {}} }
                    ---
                    failure(response, response.body.detail)
                }

            else ->
success({
          contentType: "application/json",
          status: 200,
          headers: {},
          cookies: {},
          body: {
            items: slice([
              {
                id: "1",
                name: "One",
                email: "one@acme.com"
              },
              {
                id: "2",
                name: "Two",
                email: "two@acme.com"
              },
              {
                id: "3",
                name: "Three",
                email: "three@acme.com"
              },
              {
                id: "4",
                name: "Four",
                email: "four@acme.com"
              },
            ], parameter.offset, parameter.offset + parameter.limit)
          }
    } <~ { raw: "{}" as Binary, mimeType: "application/json", transportAttributes: {kind: "http", statusCode: 200, headers: {}, cookies: {}} })
            }
}

@OperationElement()
var testFailures: Operation<{ reason: String }, HttpResponse<Any>, ResultFailure<{ reason: 'unexpected' }, UnexpectedError> | ResultFailure<{ reason: 'default' }, DefaultError> | ResultFailure<{ reason: '4xx' }, Default4XXError> | ResultFailure<{ reason: '5xx' }, Default5XXError>, HttpConnection> = {
    name: "testFailures",
    displayName: "Tests failures",
    executor: (parameter, connection) -> parameter.reason match {
        case "default" -> defaultFailure({ reason: 'default' }, {
            kind: "default",
            categories: ["default-category"]
        })
        case "4xx" -> default4XXFailure({ reason: '4xx' }, {
            kind: "4XX",
            categories: ["4xx-category"]
        })
        case "5xx" -> default5XXFailure({ reason: '5xx' }, {
            kind: "5XX",
            categories: ["5xx-category"]
        })
        case "exception" -> dw::Runtime::fail("Exception message")
        else -> unexpectedFailure({ reason: 'unexpected' }, {
            kind: "unexpected",
            categories: ["unexpected-category"]
        })
    }
}

var getAccountsPaginated: PaginatedOperation<{offset: Number,limit: Number}, {offset: Number,limit: Number}, Page<Account,
  {offset: Number, limit: Number}>, ResultFailure<HttpResponse<Any>, Error>, HttpConnection> = (getAccounts paginated (param, page) -> {
    items: page.body.items default [],
    (nextPage: {
        args: param update { case offset at .offset -> offset + sizeOf(page.body.items) }
    }) if !isEmpty(page.body.items)
})

type GetUsersRequest = {since: Number}
type GetUsersResponse = {items: Array<User>}
type User = {id: String, name: String, email: String, created_at: Number}

var getUsers : Operation<GetUsersRequest, HttpResponse<GetUsersResponse>, ResultFailure<HttpResponse, Error>, HttpConnection> = {
    name: "getUsers",
    displayName: "Get Users",
    executor: (parameter, connection) -> do {
        success({
                          contentType: "application/json",
                          status: 200,
                          headers: {},
                          cookies: {},
                          body: {
                    items: [
                      {
                          id: 1,
                          name: "pepe",
                          email: "p@sales.com",
                          created_at: 20200102
                      },
                      {
                          id: 2,
                          name: "pepito",
                          email: "pe@sales.com",
                          created_at: 20200105
                      },
                      {
                          id: 3,
                          name: "pepillo",
                          email: "pep@sales.com",
                          created_at: 20210517
                      }
                    ]
                }
                 } as HttpResponse<GetUsersResponse>)
    }
}

var getInitialDatetime : Operation<{}, Number, HttpResponse, HttpConnection> = {
    name: "getInitialDatetime",
    displayName: "Get datetime",
    executor: (parameter, connection) -> do {
        success(20200101)
    }
}

var myTriggerStrategy:TriggerStrategy<HttpResponse<GetUsersResponse>,User,User,Number> =
{
    items: (result) -> result.body.items,
    item: (item) -> item,
    watermark: (result,item) -> item.created_at,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

@TriggerElement()
var getUsersTrigger: Trigger<GetUsersRequest, GetUsersRequest, HttpResponse<GetUsersResponse>, GetUsersRequest, ResultFailure<HttpResponse, Error>, HttpConnection, User, User, Number> = {
    name: "getUsersTrigger",
    displayName: "Get Users Trigger",
    metadata: {
        order: "ASC",
        paginated: false
    },
    strategy: myTriggerStrategy,
    operation: getUsers,
    inputMapper: (ti, w) -> ti,
    initialWatermark: (GetUsersRequest, HttpConnection) -> 20200101//(now() as String { format: "uuuuMMdd" }) as Number
}

// Paginated
type GetUsersRequestPaginated = {since: Number, offset?: Number, limit: Number }

var getUsersToBePaginated : Operation<GetUsersRequestPaginated, HttpResponse<GetUsersResponse>, ResultFailure<HttpResponse, Error>, HttpConnection> = {
    name: "getUsers",
    displayName: "Get Users",
    executor: (parameter, connection) -> do {

        if (parameter.offset? == false)
        success({
                          contentType: "application/json",
                          status: 200,
                          headers: {},
                          cookies: {},
                          body: {
                    items: [
                      {
                          id: 1,
                          name: "pepe",
                          email: "p@sales.com",
                          created_at: 20200102
                      },
                      {
                          id: 2,
                          name: "pepito",
                          email: "pe@sales.com",
                          created_at: 20200105
                      },
                      {
                          id: 3,
                          name: "pepillo",
                          email: "pep@sales.com",
                          created_at: 20210517
                      }
                    ]
                }
                 } as HttpResponse<GetUsersResponse>)
         else
             if (parameter.offset == 3)
                 success({
                                   contentType: "application/json",
                                   status: 200,
                                   headers: {},
                                   cookies: {},
                                   body: {
                             items: [
                               {
                                   id: 4,
                                   name: "juan",
                                   email: "j@sales.com",
                                   created_at: 20210519
                               },
                               {
                                   id: 5,
                                   name: "perez",
                                   email: "perez@sales.com",
                                   created_at: 20211111
                               },
                               {
                                   id: 6,
                                   name: "jose",
                                   email: "jose@sales.com",
                                   created_at: 20230609
                               }
                             ]
                         }
                          } as HttpResponse<GetUsersResponse>)
              else
                      success({
                                        contentType: "application/json",
                                        status: 200,
                                        headers: {},
                                        cookies: {},
                                        body: {
                                  items: []
                              }
                               } as HttpResponse<GetUsersResponse>)
    }
}

var getUsersPaginated = offsetPaginated(
                            getUsersToBePaginated,
                             (page) -> page.body.items default [],
                             (param) -> param.offset default 0,
                             (param, from) -> (param update { case offset at .offset! -> from })
                         )

var myTriggerStrategyPaginated:TriggerStrategy<Page<User, GetUsersRequestPaginated>,User,User,Number>  =
{
    items: (result) -> result.items,
    item: (item) -> item,
    watermark: (result,item) -> item.created_at,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

//startDate is optional because for flow it should be hidden
type  CustomGetUsersRequestPaginated = {| startDate? : Number|}

@TriggerElement()
var getUsersTriggerPaginated: Trigger<CustomGetUsersRequestPaginated, GetUsersRequestPaginated, Page<User, GetUsersRequestPaginated> ,
GetUsersRequestPaginated, ResultFailure<HttpResponse, Error>, HttpConnection, User, User, Number> = {
    name: "getUsersTriggerPaginated",
    displayName: "Get Users Trigger",
    metadata: {
        order: "ASC",
        paginated: true
    },
    strategy: myTriggerStrategyPaginated,
    operation: getUsersPaginated,
    inputMapper: (ti:CustomGetUsersRequestPaginated, w:Number) -> {since: w, limit: 10 },
    initialWatermark: (triggerInput, connection) -> triggerInput.startDate default 20200101//now() as String { format: "uuuuMMdd" }) as Number
}


@TriggerElement()
var getUsersTriggerPaginatedNoStartDate: Trigger<{||}, GetUsersRequestPaginated, Page<User, GetUsersRequestPaginated> ,
GetUsersRequestPaginated, ResultFailure<HttpResponse, Error>, HttpConnection, User, User, Number> = {
    name: "getUsersTriggerPaginatedNoStartDate",
    displayName: "Get Users Trigger",
    metadata: {
        order: "ASC",
        paginated: true
    },
    strategy: myTriggerStrategyPaginated,
    operation: getUsersPaginated,
    inputMapper: (ti, w) -> {since: w, limit: 10 },
    initialWatermark: (triggerInput, connection) -> getInitialDatetime.executor({},connection) match {
                                                                                                        case is ResultSuccess -> $.value
                                                                                                        }
}
