%dw 2.7

import * from dw::core::Arrays

import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Operation
import * from com::mulesoft::connectivity::decorator::Trigger
import * from com::mulesoft::connectivity::Metadata
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::Types

// Type
type AccountUpdate = {name: String, email: String, createdAt?: LWDateTime}
type Account = AccountUpdate & { id: String}

// Connection

@OperationElement
var isAlive: Operation<{}, HttpResponse, HttpResponse, HttpConnection> = {
    name: "isAlive",
    displayName: "Is alive",
    executor: (parameter, connection) -> success(
        	{
                contentType: "application/json",
                status: 200,
                headers: {},
                cookies: {}
    		})
}

@TestConnectionElement()
var testConnection = {
	validate: defineTestConnection(
        isAlive,
        (response) -> { isValid: response.value.status == 200 }
    )
}

@ConnectionElement()
var basicAuthConnectionProvider = defineBasicHttpConnectionProvider<{baseUri: String, user: String, pass: String}>(
    (parameter) -> {username: parameter.user, password: parameter.pass},
    (parameter) -> {baseUri: parameter.baseUri}
)

@ConnectionElement()
var apiKeyAuthConnectionProvider = defineApiKeyHttpConnectionProvider<{baseUri: String, apiKey: String}>(
    (parameter) -> {apiKey: parameter.apiKey},
    (parameter) -> {baseUri: parameter.baseUri},
    {in: 'header', name: "X-API-KEY"}
)

@ConnectionElement()
var oauth2AuthCodeConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'authorizationCode',
        authorizationUrl: "http://localhost/authorize",
        tokenUrl: "http://localhost/token",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

@ConnectionElement()
var oauth2ClientCredentialsConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'clientCredentials',
        tokenUrl: "http://localhost/token",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

@ConnectionElement()
var oauth2ImplicitConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'implicit',
        authorizationUrl: "http://localhost/authorize",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

@ConnectionElement()
var oauth2PasswordConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'password',
        tokenUrl: "http://localhost/token",
        refreshUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

// Operation
@OperationElement()
var updateAccountById : Operation<{accountId: String, account: AccountUpdate}, HttpResponse<{id: String, name: String, email: String}>, HttpResponse, HttpConnection> = {
    name: "updateAccountById",
    displayName: "Update Account by Id",
    executor: (parameter, connection) -> parameter.accountId match {
        case "999" -> do {
                var response = {
                                contentType: "application/json",
                                status: 404,
                                statusText: "Not Found",
                                headers: {},
                                cookies: {},
                                body: {} <~ {mimeType: "application/json; charset=utf-8", raw: "" as Binary {base: "64"}}
                        }
                ---
                failure(response, response.body.detail)
            }
        case "-1" -> do {
                var response = {
                                contentType: "application/json",
                                status: 400,
                                headers: {},
                                cookies: {},
                                body: {
                                    error: "invalidAccountId",
                                    detail: "accountId cannot be a negative value"
                                }
                        }
                ---
                failure(response, response.body.detail)
            }
        else ->
            success({
                contentType: "application/json",
                status: 200,
                headers: {},
                cookies: {},
                body: {
                    id: parameter.accountId,
                    name: parameter.account.name,
                    email: parameter.account.email
                }
            })
        }
}

@OperationElement()
var getAccounts : Operation<{offset: Number, limit: Number}, HttpResponse<{items: Array<{id: String, name: String, email: String}>}>, HttpResponse, HttpConnection> = {
    name: "getAccounts",
    displayName: "Get Accounts",
    executor: (parameter, connection) -> parameter.offset match {
            case "-1" -> do {
                    var response = {
                                    contentType: "application/json",
                                    status: 400,
                                    statusText: "Bad Request",
                                    headers: {},
                                    cookies: {},
                                    body: {}
                            }
                    ---
                    failure(response, response.body.detail)
                }

            else ->
success({
          contentType: "application/json",
          status: 200,
          headers: {},
          cookies: {},
          body: {
            items: slice([
              {
                id: "1",
                name: "One",
                email: "one@acme.com"
              },
              {
                id: "2",
                name: "Two",
                email: "two@acme.com"
              },
              {
                id: "3",
                name: "Three",
                email: "three@acme.com"
              },
              {
                id: "4",
                name: "Four",
                email: "four@acme.com"
              },
            ], parameter.offset, parameter.offset + parameter.limit)
          }
    })
            }
}

var getAccountsPaginated: Operation<{offset: Number,limit: Number}, Page<Account,
  {offset: Number, limit: Number}>, HttpResponse<Any>, HttpConnection> = (getAccounts paginated (param, page) -> {
    items: page.body.items default [],
    (nextPage: {
        args: param update { case offset at .offset -> offset + sizeOf(page.body.items) }
    }) if !isEmpty(page.body.items)
})

type GetUsersRequest = {since: Number}
type GetUsersResponse = {items: Array<User>}
type User = {id: String, name: String, email: String, created_at: Number}

var getUsers : Operation<GetUsersRequest, HttpResponse<GetUsersResponse>, HttpResponse, HttpConnection> = {
    name: "getUsers",
    displayName: "Get Users",
    executor: (parameter, connection) -> do {
        success({
                          contentType: "application/json",
                          status: 200,
                          headers: {},
                          cookies: {},
                          body: {
                    items: [
                      {
                          id: 1,
                          name: "pepe",
                          email: "p@sales.com",
                          created_at: 20200102
                      },
                      {
                          id: 2,
                          name: "pepito",
                          email: "pe@sales.com",
                          created_at: 20200105
                      },
                      {
                          id: 3,
                          name: "pepillo",
                          email: "pep@sales.com",
                          created_at: 20210517
                      }
                    ]
                }
                 } as HttpResponse<GetUsersResponse>)
    }
}

var myTriggerStrategy:TriggerStrategy<HttpResponse<GetUsersResponse>,User,User,Number> =
{
    items: (result) -> result.body.items,
    item: (item) -> item,
    watermark: (result,item) -> item.created_at,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

@TriggerElement()
var getUsersTrigger: Trigger<GetUsersRequest, GetUsersRequest, HttpResponse<GetUsersResponse>,GetUsersRequest, HttpResponse, HttpConnection, User, User, Number> = {
    name: "getUsersTrigger",
    displayName: "Get Users Trigger",
    metadata: {
        order: "ASC",
        paginated: false
    },
    strategy: myTriggerStrategy,
    operation: getUsers,
    inputMapper: (ti, w) -> ti,
    initialWatermark: (triggerInput, connection) -> 20200101
}
