%dw 2.7

import TestConnector from Metadata
import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Annotations
import * from com::mulesoft::connectivity::Metadata
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::decorator::ValueProvider

// ------ Types Definition ------
// ----------------
//  - Auth Types -
// ----------------

type AuthConnection = @Label(value = "Salesforce basic connection") {
    orgId: @Label(value = "orgId basic connection") String,
    token: @Label(value = "token basic connection") String
}

// ----------------------------
//  - Salesforce Basic Types -
// ----------------------------

type RestApiError = {
    errorCode?: String,
    message?: String
}

type ErrorInfo = {
    message?: String,
    errorCode?: String,
    fields?: Array<String>,
    extendedErrorDetails?: Array<{ extendedErrorCode?: String }>
}

type GenericError = Array<ErrorInfo> | Array<RestApiError>

type SObjectRootInfo = {
    objectDescribe?: {
        name?: String,
        label?: String,
        labelPlural?: String,
        keyPrefix?: String,
        custom?: Boolean,
        customSetting?: Boolean,
        activateable?: Boolean,
        createable?: Boolean,
        updateable?: Boolean,
        deletable?: Boolean,
        undeletable?: Boolean,
        mergeable?: Boolean,
        replicateable?: Boolean,
        triggerable?: Boolean,
        queryable?: Boolean,
        retrieveable?: Boolean,
        searchable?: Boolean,
        layoutable?: Boolean,
        feedEnabled?: Boolean,
        mruEnabled?: Boolean,
        hasSubtypes?: Boolean,
        isSubtype?: Boolean,
        dataTranslationEnabled?: Boolean,
        isInterface?: Boolean,
        deepCloneable?: Boolean,
        associateEntityType?: String,
        associateParentEntity?: String,
        deprecatedAndHidden?: Boolean,
        urls?: {
            _?: String
        }
    },
    recentItems?: Array<{ id?: String }>
}

type SaveResult = {
    success?: Boolean,
    id?: String,
    errors?: Array<{
        message?: String,
        fields?: Array<String>,
        statusCode?: String,
        extendedErrorDetails?: Array<{
            extendedErrorCode?: String
        }>
    }>,
    warnings?: Array<{
        message?: String,
        statusCode?: String,
        extendedDetails?: Array<{ extendedErrorCode?: String }>
    }>,
    infos?: Array<{
        message?: String,
        statusCode?: String,
        extendedDetails?: Array<{ extendedErrorCode?: String }>
    }>
}

// -----------------------------
//    - Salesforce VP Types -
// -----------------------------

type Picklist = {|
    active: Boolean,
    defaultValue: Boolean,
    label: String,
    validFor: Object,
    value: String
|}

type RecordIdName = {|
    Id: String,
    Name: String
|}

// -----------------------------
//  - Salesforce Record Types -
// -----------------------------

type AccountSObject = {
    id?: String,
    isDeleted?: Boolean,
    masterRecordId?: String,
    name?: String,
    "type"?: String,
    parentId?: String,
    billingStreet?: String,
    billingCity?: String,
    billingState?: String,
    billingPostalCode?: String,
    billingCountry?: String,
    billingLatitude?: Number,
    billingLongitude?: Number,
    billingGeocodeAccuracy?: String,
    billingAddress?: Object,
    shippingStreet?: String,
    shippingCity?: String,
    shippingState?: String,
    shippingPostalCode?: String,
    shippingCountry?: String,
    shippingLatitude?: Number,
    shippingLongitude?: Number,
    shippingGeocodeAccuracy?: String,
    shippingAddress?: Object,
    phone?: String,
    fax?: String,
    accountNumber?: String,
    website?: String,
    photoUrl?: String,
    sic?: String,
    industry?: String,
    annualRevenue?: Number,
    numberOfEmployees?: Number,
    ownership?: String,
    tickerSymbol?: String,
    description?: String,
    rating?: String,
    site?: String,
    ownerId?: String,
    createdDate?: DateTime,
    createdById?: String,
    lastModifiedDate?: DateTime,
    lastModifiedById?: String,
    systemModstamp?: DateTime,
    lastActivityDate?: Date,
    lastViewedDate?: DateTime,
    lastReferencedDate?: DateTime,
    jigsaw?: String,
    jigsawCompanyId?: String,
    cleanStatus?: String,
    accountSource?: String,
    dunsNumber?: String,
    tradestyle?: String,
    naicsCode?: String,
    naicsDesc?: String,
    yearStarted?: String,
    sicDesc?: String,
    dandbCompanyId?: String,
    attributes?: {
        "type"?: String,
        url?: String
    }
}

type AccountPostPatchSObject = {
    // TODO (juldiazmule): value is defaulted at creation, should we still allow its modification
    // OwnerId: @ValuesFrom(value = { name: "userIdsValueProvider" } ) String,
    billingGeocodeAccuracy: @Label(value = "Billing Geocode Accuracy") @ValuesFrom(value = { name: "accountBillingGeocodeAccuracyValueProvider" }) String,
    name?: @Label(value = "Account Name") String,
    accountNumber?: @Label(value = "Account Number") String,
    accountSource?: @ValuesFrom(value = { name: "accountAccountSourceValueProvider" }) String,
    annualRevenue?: @Label(value = "Annual Revenue") Number,
    billingCity?: @Label(value = "Billing City") String,
    billingCountry?: @Label(value = "Billing Country") String,
    billingLatitude?: @Label(value = "Billing Latitude") Number,
    billingLongitude?: @Label(value = "Billing Longitude") Number,
    billingPostalCode?: @Label(value = "Billing Zip/Postal Code") String,
    billingState?: @Label(value = "Billing State/Province") String,
    billingStreet?: @Label(value = "Billing Street") String,
    cleanStatus?: @Label(value = "Clean Status") @ValuesFrom(value = { name: "accountCleanStatusValueProvider" }) String,
    dandbCompanyId?: @Label(value = "D&B Company ID") @ValuesFrom(value = { name: "accountDandBCompanyIdsValueProvider" }) String,
    description?: @Label(value = "Description") String,
    dunsNumber?: @Label(value = "D-U-N-S Number") String,
    fax?: @Label(value = "Account Fax") String,
    industry?: @Label(value = "Industry") @ValuesFrom(value = { name: "accountIndustryValueProvider" }) String,
    jigsaw?: @Label(value = "Data.com Key") String,
    naicsCode?: @Label(value = "NAICS Code") String,
    naicsDesc?: @Label(value = "AICS Description") String,
    numberOfEmployees?: @Label(value = "Employees") Number,
    operatingHoursId?: @Label(value = "Operating Hour ID") @ValuesFrom(value = { name: "accountOperatingHoursIdsValueProvider" }) String,
    ownership?: @Label(value = "Ownership") @ValuesFrom(value = { name: "accountOwnershipValueProvider" }) String,
    parentId?: @Label(value = "Parent Account ID") @ValuesFrom(value = { name: "accountIdsValueProvider" }) String,
    phone?: @Label(value = "Account Phone") String,
    rating?: @Label(value = "Account Rating") @ValuesFrom(value = { name: "accountRatingValueProvider" }) String,
    shippingCity?: @Label(value = "Shipping City") String,
    shippingCountry?: @Label(value = "Shipping Country") String,
    shippingGeocodeAccuracy?: @Label(value = "Shipping Geocode Accuracy") @ValuesFrom(value = { name: "accountShippingGeocodeAccuracyValueProvider" }) String,
    shippingLatitude?: @Label(value = "Shipping Latitude") Number,
    shippingLongitude?: @Label(value = "Shipping Longitude") Number,
    shippingPostalCode?: @Label(value = "Shipping Zip/Postal Code") String,
    shippingState?: @Label(value = "Shipping State/Province") String,
    shippingStreet?: @Label(value = "Shipping Street") String,
    sic?: @Label(value = "SIC Code") String,
    sicDesc?: @Label(value = "SIC Description") String,
    site?: @Label(value = "Account Site") String,
    tickerSymbol?: @Label(value = "Ticker Symbol") String,
    tradestyle?: @Label(value = "Tradestyle") String,
    "type"?: @Label(value = "Account Type") @ValuesFrom(value = { name: "accountTypeValueProvider" }) String,
    website?: @Label(value = "Website") String,
    yearStarted?: @Label(value = "Year Started") String,
}

// --------------------------------
//  - Salesforce Operation Types -
// --------------------------------

type AccountPostOperation = {
    response400: Array<ErrorInfo>,
    response401: Array<ErrorInfo>,
    response403: Array<RestApiError>,
    response404: Array<RestApiError>,
    response500: Array<ErrorInfo>,
    response405: Array<RestApiError>,
    response503: Array<RestApiError>,
    response201: SaveResult,
    response415: Array<ErrorInfo>,
    request: HttpRequestType<{|
        query?: Object,
        headers?: {
            "Content-Type"?: String
        },
        cookie?: Object,
        body: AccountPostPatchSObject
    |}>,
    response: SaveResult
}

type AccountGetOperation = {
    response304: Array<RestApiError>,
    response400: Array<ErrorInfo>,
    response401: Array<ErrorInfo>,
    response403: Array<RestApiError>,
    response404: Array<RestApiError>,
    response500: Array<ErrorInfo>,
    response405: Array<RestApiError>,
    response503: Array<RestApiError>,
    response200: AccountSObject,
    response412: Array<RestApiError>,
    request: HttpRequestType<{|
        uri: {| id: String |},
        query?: {
            fields?: String
        },
        headers?: {
            "If-Modified-Since"?: String,
            "If-Unmodified-Since"?: String,
            "If-Match"?: String,
            "If-None-Match"?: String
        },
        cookie?: Object
    |}>,
    response: AccountSObject
}

type AccountDeleteOperation = {
    response400: Array<ErrorInfo>,
    response401: Array<ErrorInfo>,
    response403: Array<RestApiError>,
    response404: Array<RestApiError>,
    response500: Array<ErrorInfo>,
    response405: Array<RestApiError>,
    response503: Array<RestApiError>,
    response204: Any,
    response415: Array<ErrorInfo>,
    request: HttpRequestType<{|
        uri: {| id: String |},
        query?: Object,
        headers?: Object,
        cookie?: Object
    |}>,
    response: Any
}

type AccountPatchOperation = {
    response400: Array<ErrorInfo>,
    response401: Array<ErrorInfo>,
    response403: Array<RestApiError>,
    response404: Array<RestApiError>,
    response500: Array<ErrorInfo>,
    response405: Array<RestApiError>,
    response503: Array<RestApiError>,
    response204: Any,
    response415: Array<ErrorInfo>,
    request: HttpRequestType<{|
        uri: {| id: String |},
        query?: Object,
        headers?: {
            "Content-Type"?: String
        },
        cookie?: Object,
        body: AccountPostPatchSObject
    |}>,
    response: Any
}

// ----------------------------
//  - VP Retriever Functions -
// ----------------------------

fun picklistValueProviderExecutor(path: String, field: String): Executor<{}, Array<Picklist>, ResultFailure<Any, Error>, HttpConnection> =
	(parameter, connection) -> do {
	    var providedValues = connection({
            method: "GET",
            path: "/sobjects/$(path)/describe",
            queryParams: {},
            headers: {},
            cookie: {}
        })
        ---
        if (providedValues.status == 200)
            success((providedValues.body.fields filter ($.name == field))[0].picklistValues as Array<Picklist>)
        else
            failure(providedValues)
    }

fun recordIdsValueProviderExecutor(recordType: String): Executor<{}, Array<RecordIdName>, ResultFailure<Any, Error>, HttpConnection> =
	(parameter, connection) -> do {
	    var providedValues = connection({
            method: "GET",
            path: "/query?q=SELECT Id, Name FROM $(recordType) ORDER BY Name",
            // TODO (juldiazmule): above should be bellow but I'm not so sure for this particular case
            queryParams: {},
            headers: {},
            cookie: {}
        })
        ---
        if (providedValues.status == 200)
            success(providedValues.body.records as Array<RecordIdName>)
        else
            failure(providedValues)
    }

// ---------------------
//  - Record Ids VPs -
// ---------------------

var accountIdsValueProvider = defineValueProvider(
    recordIdsValueProviderExecutor("ACCOUNT"),
    (response: Array<RecordIdName>) -> response map {value: $.Id, displayValue: {label: $.Name}}
)

var userIdsValueProvider = defineValueProvider(
    recordIdsValueProviderExecutor("USER"),
    (response: Array<RecordIdName>) -> response map {value: $.Id, displayValue: {label: $.Name}}
)

// -----------------
//  - Account VPs -
// -----------------

var accountAccountSourceValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "AccountSource"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

var accountBillingGeocodeAccuracyValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "BillingGeocodeAccuracy"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

var accountCleanStatusValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "CleanStatus"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

var accountDandBCompanyIdsValueProvider = defineValueProvider(
    recordIdsValueProviderExecutor("DandBCompany"),
    (response: Array<RecordIdName>) -> response map {value: $.Id, displayValue: {label: $.Name}}
)

var accountIndustryValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "Industry"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

var accountOperatingHoursIdsValueProvider = defineValueProvider(
    recordIdsValueProviderExecutor("OperatingHours"),
    (response: Array<RecordIdName>) -> response map {value: $.Id, displayValue: {label: $.Name}}
)

var accountOwnershipValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "Ownership"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

var accountRatingValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "Rating"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

var accountShippingGeocodeAccuracyValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "ShippingGeocodeAccuracy"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

var accountTypeValueProvider = defineValueProvider(
    picklistValueProviderExecutor("Account", "Type"),
    (response: Array<Picklist>) -> response map {value: $.value, displayValue: {label: $.label}}
)

// ------ Module Definition ------
var SALESFORCE_BASE_URL = 'https://{orgId}.salesforce.com/services/data/v60.0'

@ConnectionElement()
var salesforceBearerConnection = defineBearerHttpConnectionProvider<AuthConnection>(
    (parameter) -> { token: parameter.token },
    (parameter) -> {
        baseUri: SALESFORCE_BASE_URL replace "{orgId}" with parameter.orgId
    }
)

var getAccountMetadataForTestConnection : Operation<{}, Any, ResultFailure<Any, Error>, HttpConnection> = {
    name: "getAccountMetadataForTestConnection",
    displayName: "Get Account Metadata For TestConnection",
    executor: (parameter, connection) -> do {
        var response = connection({
            method: 'GET',
            path: '/sobjects/Account/describe'
        })
        ---
        if (response.status == 200)
            success(response)
        else
            failure(response)
    }
}

@TestConnectionElement()
var testConnection : HttpTestConnection = {
    validate: defineTestConnection(
        getAccountMetadataForTestConnection,
        (response) -> {
            isValid: response.success,
            message: if (response.success) "Connection test succeeded" else "Connection test failed"
        }
    )
}

var operations = {
    "/Account": {
        post: {
            name: "createAccount",
            displayName: "Create Account",
            executor: (parameter, connection) -> do {
                var response = connection({
                    method: "POST",
                    path: "/sobjects/Account",
                    queryParams: parameter.query default {},
                    headers: parameter.headers default {},
                    cookie: parameter.cookie default {},
                    body: parameter.body default {}
                })
                ---
                if (response.status == 201)
                    success(response)
                else
                    failure(response)
            }
        } as HttpOperation<
            AccountPostOperation.request,
            AccountPostOperation.response,
            GenericError,
            HttpConnection
        >
    },

    "/Account/{id}": {
        get: {
            name: "getAccountById",
            displayName: "Get Account By Id",
            executor: (parameter, connection) -> do {
                var response = connection({
                    method: "GET",
                    path: "/sobjects/Account/" ++ parameter.uri.id,
                    queryParams: parameter.query default {},
                    headers: parameter.headers default {},
                    cookie: parameter.cookie default {},
                })
                ---
                if (response.status == 200)
                    success(response)
                else
                    failure(response)
            }
        } as HttpOperation<
            AccountGetOperation.request,
            AccountGetOperation.response,
            GenericError,
            HttpConnection
        >,
        delete: {
            name: "deleteAccountById",
            displayName: "Delete Account By Id",
            executor: (parameter, connection) -> do {
                var response = connection({
                    method: "DELETE",
                    path: "/sobjects/Account/" ++ parameter.uri.id,
                    queryParams: parameter.query default {},
                    headers: parameter.headers default {},
                    cookie: parameter.cookie default {}
                })
                ---
                if (response.status == 204)
                    success(response)
                else
                    failure(response)
            }
        } as HttpOperation<
            AccountDeleteOperation.request,
            AccountDeleteOperation.response,
            GenericError,
            HttpConnection
        >,
        patch: {
            name: "updateAccountById",
            displayName: "Update Account By Id",
            executor: (parameter, connection) -> do {
                var response = connection({
                    method: "PATCH",
                    path: "/sobjects/Account/" ++ parameter.uri.id,
                    queryParams: parameter.query default {},
                    headers: parameter.headers default {},
                    cookie: parameter.cookie default {},
                    body: parameter.body default {}
                })
                ---
                if (response.status == 204)
                    success(response)
                else
                    failure(response)
            }
        } as HttpOperation<
            AccountPatchOperation.request,
            AccountPatchOperation.response,
            GenericError,
            HttpConnection
        >
    },
}

// ------ Connector Definition ------
@TestConnector()
var connector = {
    name: "SALESFORCE",
    displayName: "Salesforce",
    version: "0.1.0",
    releaseStatus: "PILOT",
    description: "Salesforce Connector",
    vendor: "Salesforce",
    icons: [
        {
            name: "Salesforce",
            alternateText: "Salesforce Connector",
            resource: "icon/salesforce.svg",
            size: 13040,
            dimensions: "256x180"
        }
    ],
    connections: {
      salesforceBearerConnection: salesforceBearerConnection
    },
    testConnection: testConnection,
    operations: {
        getAccount: operations."/Account/{id}".get as HttpOperation<AccountGetOperation.request,AccountGetOperation.response,GenericError,HttpConnection>,
        createAccount: operations."/Account".post as HttpOperation<AccountPostOperation.request, AccountPostOperation.response, GenericError,HttpConnection>,
        updateAccount: operations."/Account/{id}".patch as HttpOperation<AccountPatchOperation.request, AccountPatchOperation.response, GenericError,HttpConnection>,
    },
    valueProviders: {
        accountAccountSourceValueProvider: accountAccountSourceValueProvider,
        accountBillingGeocodeAccuracyValueProvider: accountBillingGeocodeAccuracyValueProvider,
        accountCleanStatusValueProvider: accountCleanStatusValueProvider,
        accountDandBCompanyIdsValueProvider: accountDandBCompanyIdsValueProvider,
        accountIdsValueProvider: accountIdsValueProvider,
        accountIndustryValueProvider: accountIndustryValueProvider,
        accountOperatingHoursIdsValueProvider: accountOperatingHoursIdsValueProvider,
        accountOwnershipValueProvider: accountOwnershipValueProvider,
        accountRatingValueProvider: accountRatingValueProvider,
        accountShippingGeocodeAccuracyValueProvider: accountShippingGeocodeAccuracyValueProvider,
        accountTypeValueProvider: accountTypeValueProvider,
        userIdsValueProvider: userIdsValueProvider
    },
    metadataProviders: {}
}
