/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.model;

import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.CATEGORY_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.CONNECTION_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.DESCRIPTION_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.DISPLAY_NAME_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.MODEL_REFERENCE_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.NAME_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.OPERATION_OR_TRIGGER_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.TESTCONNECTION_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.VENDOR_REQUIRED;
import static com.mulesoft.connectivity.mule.internal.model.ValidationMessages.VERSION_REQUIRED;

import org.mule.runtime.api.meta.Category;

import com.mulesoft.connectivity.linkweave.api.model.ConnectorModel;
import com.mulesoft.connectivity.linkweave.api.model.connection.TestConnectionModel;
import com.mulesoft.connectivity.linkweave.api.model.operation.OperationModel;
import com.mulesoft.connectivity.linkweave.api.model.provider.ValueProviderModel;
import com.mulesoft.connectivity.linkweave.api.model.trigger.TriggerModel;

import java.io.Serializable;
import java.util.List;

import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.NullUnmarked;

/**
 * A Java Contract for the Mule Connector. Please refer to the Builder for more details.
 */
@NullUnmarked
public class MuleConnectorModel extends ConnectorModel implements Serializable {

  private static final long serialVersionUID = 1L;

  /**
   * Connector's Version in SemVer format. Required.
   */
  private final String version;

  /**
   * Connector's Description. Required.
   */
  private final String description;

  /**
   * Connector's Vendor (The namespace, for example, Standard vs. Custom). Required.
   */
  private final String vendor;

  /**
   * Connector's Category. Required.
   */
  private final Category category;

  /**
   * List of Connection Providers. Required.
   */
  private final List<MuleConnectionProviderModel> connectionProviderModelList;

  /**
   * Test connection Model.
   */
  private final TestConnectionModel testConnectionModel;

  private final List<OperationModel> operations;
  /**
   * List of triggers.
   */
  private final List<TriggerModel> triggers;


  private final List<ValueProviderModel> valueProviders;

  private MuleConnectorModel(String name, String displayName, String version, String description,
                             String vendor, Category category, List<MuleConnectionProviderModel> connectionProviderModelList,
                             TestConnectionModel testConnectionModel, List<OperationModel> operations,
                             List<TriggerModel> triggers,
                             List<ValueProviderModel> valueProviders,
                             String modelReference) {
    super(modelReference, name, displayName);
    this.version = version;
    this.description = description;
    this.vendor = vendor;
    this.category = category;
    this.connectionProviderModelList = connectionProviderModelList;
    this.testConnectionModel = testConnectionModel;
    this.operations = operations;
    this.triggers = triggers;
    this.valueProviders = valueProviders;
  }

  /**
   * Gets the Connector's Version.
   *
   * @return Connector's Version in SemVer format.
   */
  public String getVersion() {
    return this.version;
  }

  /**
   * Gets the Connector's Category.
   *
   * @return Connector Category
   */
  public Category getCategory() {
    return this.category;
  }

  /**
   * Gets the Connector's Description.
   *
   * @return String with the Connector Description
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Gets the connector's vendor.
   *
   * @return Connector's vendor.
   */
  public String getVendor() {
    return this.vendor;
  }

  /**
   * Gets the Connector's connection definitions.
   *
   * @return list of connection models.
   */
  public List<MuleConnectionProviderModel> getConnections() {
    return List.copyOf(this.connectionProviderModelList);
  }

  /**
   * Gets the test connection model.
   *
   * @return test connection definition.
   */
  public TestConnectionModel getTestConnection() {
    return this.testConnectionModel;
  }

  /**
   * Gets the list of valueProvider models.
   *
   * @return list of value provider definitions.
   */
  public List<ValueProviderModel> getValueProviders() {
    return valueProviders;
  }

  /**
   * Gets the value provider model based on value provider reference.
   *
   * @param valueProviderReference value provider reference
   * @return value provider model
   */
  public ValueProviderModel getValueProvider(String valueProviderReference) {

    return this.getValueProviders().stream()
        .filter(provider -> provider.getName().equals(valueProviderReference))
        .findFirst()
        .orElseThrow(() -> new IllegalStateException("Value provider not found: " + valueProviderReference));
  }

  /**
   * Gets the list of operation models.
   *
   * @return list of operation definitions.
   */
  public List<OperationModel> getOperations() {
    return this.operations;
  }

  /**
   * Gets the list of trigger models.
   *
   * @return list of trigger definitions.
   */
  public List<TriggerModel> getTriggers() {
    return this.triggers;
  }

  @Override
  public String toString() {
    return "MuleConnectorModel{" +
        "name='" + getName() + '\'' +
        "displayName='" + getDisplayName() + '\'' +
        "version='" + version + '\'' +
        "description='" + description + '\'' +
        "vendor='" + vendor + '\'' +
        "modelReference='" + modelReference + '\'' +
        "category=" + category + '\'' +
        "connections='" + connectionProviderModelList + '\'' +
        "testConnection='" + testConnectionModel + '\'' +
        "operations='" + operations + '\'' +
        "triggers='" + triggers + '\'' +
        '}';
  }

  /**
   * Builds MuleConnectorModel Instances.
   */
  public static class Builder extends ConnectorModel.Builder<@NonNull MuleConnectorModel, @NonNull Builder> {

    private String version;
    private String description;
    private String vendor;
    private Category category;
    private List<MuleConnectionProviderModel> connectionProviderModelList = List.of();
    private TestConnectionModel testConnectionModel;
    private List<OperationModel> operations = List.of();
    private List<TriggerModel> triggers = List.of();
    private List<ValueProviderModel> valueProviders = List.of();

    @Override
    protected MuleConnectorModel createInstance() {
      validate();
      return new MuleConnectorModel(name,
                                    displayName,
                                    version,
                                    description,
                                    vendor,
                                    category,
                                    connectionProviderModelList,
                                    testConnectionModel,
                                    operations,
                                    triggers,
                                    valueProviders,
                                    modelReference);
    }

    @Override
    protected Builder getThis() {
      return this;
    }

    /**
     * Supplies the Connector's version. (required)
     *
     * @param version String with the connector's version using semVer format.
     * @return the Builder instance
     */
    public Builder withVersion(String version) {
      this.version = version;
      return this;
    }

    /**
     * Supplies the Connector's Category. (required)
     *
     * @param category Connector's category.
     * @return the Builder instance.
     */
    public Builder withCategory(Category category) {
      this.category = category;
      return this;
    }

    /**
     * Supplies the Connector's description. (required)
     *
     * @param description Connector's description.
     * @return the Builder instance.
     */
    public Builder withDescription(String description) {
      this.description = description;
      return this;
    }

    /**
     * Supplies the Connector's vendor. (required)
     *
     * @param vendor the Connector's vendor.
     * @return the Builder instance.
     */
    public Builder withVendor(String vendor) {
      this.vendor = vendor;
      return this;
    }

    /**
     * Supplies the Connector's Connection provider models.
     *
     * @param connectionProviderModelList List of connection definitions.
     * @return the Builder instance.
     */
    public Builder withConnections(List<MuleConnectionProviderModel> connectionProviderModelList) {
      this.connectionProviderModelList = connectionProviderModelList;
      return this;
    }

    /**
     * Supplies the Connector's test connection model.
     *
     * @param testConnectionModel TestConnection Model for the Connection.
     * @return the Builder Instance.
     */
    public Builder withTestConnection(TestConnectionModel testConnectionModel) {
      this.testConnectionModel = testConnectionModel;
      return this;
    }

    /**
     * Supplies the Connector's operation models.
     *
     * @param operations List of operation definitions
     * @return the Builder instance.
     */
    public Builder withOperations(List<OperationModel> operations) {
      this.operations = operations;
      return this;
    }

    /**
     * Supplies the Connector's trigger models.
     *
     * @param triggers List of trigger definitions
     * @return the Builder instance.
     */
    public Builder withTriggers(List<TriggerModel> triggers) {
      this.triggers = triggers;
      return this;
    }

    /**
     * Supplies the Connector's value provider models.
     *
     * @param valueProviders List of value provider definitions
     * @return the Builder instance.
     */
    public Builder withValueProviders(List<ValueProviderModel> valueProviders) {
      this.valueProviders = valueProviders;
      return this;
    }

    /**
     *
     * Validates the Builder's parameters.
     *
     * @throws IllegalStateException whenever there's an invalid configuration.
     */
    private void validate() throws IllegalStateException {
      StringBuilder messages = new StringBuilder();
      if (name == null || name.isBlank()) {
        messages.append(NAME_REQUIRED).append("\n");
      }
      if (displayName == null || displayName.isBlank()) {
        messages.append(DISPLAY_NAME_REQUIRED).append("\n");
      }
      if (version == null || version.isBlank()) {
        messages.append(VERSION_REQUIRED).append("\n");
      }
      if (description == null || description.isBlank()) {
        messages.append(DESCRIPTION_REQUIRED).append("\n");
      }
      if (vendor == null || vendor.isBlank()) {
        messages.append(VENDOR_REQUIRED).append("\n");
      }
      if (this.category == null) {
        messages.append(CATEGORY_REQUIRED).append("\n");
      }
      if (modelReference == null || modelReference.isBlank()) {
        messages.append(MODEL_REFERENCE_REQUIRED).append("\n");
      }
      if (isEmptyOrNull(connectionProviderModelList)) {
        messages.append(CONNECTION_REQUIRED).append("\n");
      }
      if (this.testConnectionModel == null) {
        messages.append(TESTCONNECTION_REQUIRED).append("\n");
      }
      validateOperationsAndTriggers(messages);
      if (!messages.isEmpty()) {
        throw new IllegalStateException(messages.toString());
      }
    }

    private boolean isEmptyOrNull(List<?> list) {
      return list == null || list.isEmpty();
    }

    private void validateOperationsAndTriggers(StringBuilder messages) {
      if (isEmptyOrNull(this.operations) && isEmptyOrNull(this.triggers)) {
        messages.append(OPERATION_OR_TRIGGER_REQUIRED).append("\n");
      }
    }

  }
}
