/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.persistence;

import static org.mule.metadata.persistence.MetadataTypeConstants.OBJECT;

import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.CATALOG_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.CATEGORY_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.CONNECTIONS_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.DESCRIPTION_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.DISPLAY_NAME_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.MODEL_REFERENCE_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.NAME_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.OPERATIONS_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.SOURCES_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.TEST_CONNECTION_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.VALUE_PROVIDERS_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.VENDOR_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.VERSION_PROPERTY;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.MetadataSerializingException;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;
import org.mule.metadata.persistence.serializer.ObjectTypeSerializer;
import org.mule.metadata.persistence.serializer.TypeSerializerVisitor;

import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Stack;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.internal.bind.JsonTreeWriter;

/**
 * Gson based Type Adapter for serializing {@link MuleConnectorSerializableModel}.
 */
public class MuleConnectorSerializableModelTypeAdapter implements JsonSerializer<MuleConnectorSerializableModel> {

  private final CatalogTypeCollector catalogCollector;

  public MuleConnectorSerializableModelTypeAdapter(CatalogTypeCollector catalogCollector) {
    this.catalogCollector = catalogCollector;
  }

  @Override
  public JsonElement serialize(MuleConnectorSerializableModel src, Type typeOfSrc, JsonSerializationContext context) {
    JsonObject result = new JsonObject();
    result.addProperty(NAME_PROPERTY, src.getName());
    result.addProperty(DISPLAY_NAME_PROPERTY, src.getDisplayName());
    result.addProperty(VERSION_PROPERTY, src.getVersion());
    result.addProperty(CATEGORY_PROPERTY, src.getCategory().toString());
    result.addProperty(DESCRIPTION_PROPERTY, src.getDescription());
    result.addProperty(VENDOR_PROPERTY, src.getVendor());
    String modelReference = src.getModelReference()
        .orElseThrow(() -> new NoSuchElementException("Connector model does not have a DataWeave model reference!"));
    result.addProperty(MODEL_REFERENCE_PROPERTY, modelReference);
    result.add(CONNECTIONS_PROPERTY, context.serialize(src.getConnections()));
    result.add(TEST_CONNECTION_PROPERTY, context.serialize(src.getTestConnection()));
    result.add(OPERATIONS_PROPERTY, context.serialize(src.getOperations()));
    result.add(SOURCES_PROPERTY, context.serialize(src.getSources()));
    result.add(VALUE_PROVIDERS_PROPERTY, context.serialize(src.getValueProviders()));
    result.add(CATALOG_PROPERTY, serializeCatalog());
    return result;
  }

  private JsonElement serializeCatalog() {
    Stack<MetadataType> typeStack = new Stack<>();
    Map<String, ObjectType> registeredTypes = catalogCollector.getCatalog();
    ObjectTypeReferenceHandler referenceHandler = new ExternalCatalogObjectTypeReferenceHandler(catalogCollector);
    try (JsonTreeWriter writer = new JsonTreeWriter()) {
      writer.beginObject();
      registeredTypes.forEach((id, type) -> {
        try {
          writer.name(id);
          TypeSerializerVisitor visitor = new TypeSerializerVisitor(writer, referenceHandler, new Stack<>(), false);
          new ObjectTypeSerializer(visitor).serialize(writer, type, typeStack);
        } catch (Exception e) {
          throw new MetadataSerializingException(OBJECT + " MetadataType for catalog", e);
        }
      });
      writer.endObject();
      writer.flush();
      return writer.get();
    } catch (IOException e) {
      throw new MetadataSerializingException("Failed to serialize catalog", e);
    }
  }

}
