/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.utils;

import com.mulesoft.connectivity.linkweave.api.model.ExecutableComponentModel;
import com.mulesoft.connectivity.linkweave.api.model.provider.ContextReferenceVariable;
import com.mulesoft.connectivity.linkweave.api.model.provider.ObjectFieldSelector;
import com.mulesoft.connectivity.linkweave.api.model.provider.ProviderArgument;
import com.mulesoft.connectivity.linkweave.api.model.provider.ProviderReference;
import com.mulesoft.connectivity.linkweave.api.model.provider.TypeReferenceExpression;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedContextReferenceVariable;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedObjectFieldSelector;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedProviderArgument;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedProviderReference;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedProviderReferences;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedTypeReferenceExpression;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Transforms model-api provider models to serialized classes for persistence.
 */
public class ModelAPIToSerializedTransformer {

  /**
   * Private constructor to prevent instantiation of utility class.
   */
  private ModelAPIToSerializedTransformer() {
    // Utility class - no instantiation needed
  }

  /**
   * Transforms resolved providers from model-api to serialized classes.
   *
   * @param modelApiProviders the map from model-api models
   * @return the serialized providers map
   */
  public static Map<SerializedObjectFieldSelector, SerializedProviderReferences> transformResolvedProviders(
                                                                                                            Map<ObjectFieldSelector, ExecutableComponentModel.ProviderReferences> modelApiProviders) {

    return modelApiProviders.entrySet().stream()
        .collect(Collectors.toMap(
                                  entry -> transformObjectFieldSelector(entry.getKey()),
                                  entry -> transformProviderReferences(entry.getValue())));
  }

  /**
   * Transforms model-api ProviderReferences to serialized ProviderReferences.
   *
   * @param modelApiProviderReferences the model-api provider references to transform
   * @return the serialized provider references
   */
  private static SerializedProviderReferences transformProviderReferences(
                                                                          ExecutableComponentModel.ProviderReferences modelApiProviderReferences) {

    return new SerializedProviderReferences(
                                            transformProviderReference(modelApiProviderReferences.getValueProviderReference()),
                                            transformProviderReference(modelApiProviderReferences
                                                .getMetadataProviderReference()));
  }

  /**
   * Transforms a model-api ProviderReference to a serialized ProviderReference.
   *
   * @param modelApiProviderReference the model-api provider reference to transform
   * @return the serialized provider reference, or null if input is null
   */
  private static SerializedProviderReference transformProviderReference(
                                                                        ProviderReference modelApiProviderReference) {

    if (modelApiProviderReference == null) {
      return null;
    }

    List<SerializedProviderArgument> arguments =
        modelApiProviderReference.getArguments().stream()
            .map(ModelAPIToSerializedTransformer::transformProviderArgument)
            .collect(Collectors.toList());

    return new SerializedProviderReference(
                                           modelApiProviderReference.getName(),
                                           arguments);
  }

  /**
   * Transforms a model-api ProviderArgument to a serialized ProviderArgument.
   *
   * @param modelApiProviderArgument the model-api provider argument to transform
   * @return the serialized provider argument
   */
  private static SerializedProviderArgument transformProviderArgument(
                                                                      ProviderArgument modelApiProviderArgument) {

    return new SerializedProviderArgument(
                                          transformObjectFieldSelector(modelApiProviderArgument.getParameterSelector()),
                                          transformTypeReferenceExpression(modelApiProviderArgument.getInputSelector()));
  }

  /**
   * Transforms a model-api ObjectFieldSelector to a serialized ObjectFieldSelector.
   *
   * @param modelApiSelector the model-api object field selector to transform
   * @return the serialized object field selector
   */
  private static SerializedObjectFieldSelector transformObjectFieldSelector(
                                                                            ObjectFieldSelector modelApiSelector) {
    // not handling nested structures for ProviderReferences (not supported by Mule)
    if (modelApiSelector.getPath().length > 1) {
      throw new RuntimeException("Nested Providers not allowed: " + String.join(".", modelApiSelector.getPath()));
    }

    return new SerializedObjectFieldSelector(
                                             modelApiSelector.isRelative(),
                                             modelApiSelector.getPath());
  }

  /**
   * Transforms a model-api TypeReferenceExpression to a serialized TypeReferenceExpression.
   *
   * @param modelApiTypeRef the model-api type reference expression to transform
   * @return the serialized type reference expression
   * @throws IllegalArgumentException if the type reference expression is not supported
   */
  private static SerializedTypeReferenceExpression transformTypeReferenceExpression(
                                                                                    TypeReferenceExpression modelApiTypeRef) {

    if (modelApiTypeRef instanceof ContextReferenceVariable contextVar) {
      return new SerializedContextReferenceVariable(contextVar.getName());
    } else if (modelApiTypeRef instanceof ObjectFieldSelector objectFieldSelector) {
      return new SerializedObjectFieldSelector(objectFieldSelector.isRelative(), objectFieldSelector.getPath());
    } else {
      throw new IllegalArgumentException("Unknown type reference expression: " + modelApiTypeRef.getClass());
    }
  }
}
