/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.utils;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.TypeParameterType;
import org.mule.runtime.api.meta.Category;
import org.mule.weave.v2.ts.TypeParameter;
import org.mule.weave.v2.ts.WeaveType;

import com.mulesoft.connectivity.linkweave.api.model.connection.TestConnectionModel;
import com.mulesoft.connectivity.linkweave.api.model.operation.ErrorModel;
import com.mulesoft.connectivity.linkweave.api.model.operation.OperationModel;
import com.mulesoft.connectivity.linkweave.api.model.provider.ValueProviderModel;
import com.mulesoft.connectivity.linkweave.api.model.trigger.TriggerModel;
import com.mulesoft.connectivity.mule.internal.model.MuleConnectionProviderModel;
import com.mulesoft.connectivity.mule.internal.model.MuleConnectorModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleErrorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleOperationSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleSourceSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleValueProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.ApiKeyHttpAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.HttpAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.MuleConnectionProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.MuleTestConnectionSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.oauth.OAuth2AuthCodeAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.oauth.OAuth2ClientCredentialsAuthenticationType;

import java.util.NoSuchElementException;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Transforms {@link MuleConnectorModel} having {@link com.mulesoft.connectivity.linkweave.api.model.TypeModel} into
 * {@link MuleConnectorSerializableModel} containing {@link org.mule.metadata.api.model.MetadataType}.
 */
public class MuleConnectorModelTransformer {

  private final MuleTypeTransformer muleTypeTransformer;
  private static final Set<String> CUSTOM_INPUT_FIELDS = Set.of("customHeaders", "customQueryParams");
  private static final String GENERIC_ERROR_TYPE_FIELD = "ErrorType";

  public MuleConnectorModelTransformer() {
    muleTypeTransformer = new MuleTypeTransformer();
  }

  private Supplier<? extends RuntimeException> shouldHaveRef(String kind) {
    return () -> new NoSuchElementException("Every " + kind + " should have a model reference");
  }

  /**
   * Transforms the {@link MuleConnectorModel} into {@link MuleConnectorSerializableModel}.
   *
   * @param muleConnectorModel to be converted
   * @return Converted object of {@link MuleConnectorSerializableModel}
   */
  public MuleConnectorSerializableModel toSerializableModel(MuleConnectorModel muleConnectorModel) {
    return MuleConnectorSerializableModel.builder()
        .name(muleConnectorModel.getName())
        .displayName(muleConnectorModel.getDisplayName())
        .version(muleConnectorModel.getVersion())
        .description(muleConnectorModel.getDescription())
        .vendor(muleConnectorModel.getVendor())
        .category(Category.valueOf(muleConnectorModel.getCategory().name()))
        .connectionProviders(muleConnectorModel.getConnections().stream()
            .map(this::toSerializableConnectionProvider).toList())
        .testConnection(toSerializableTestConnection(muleConnectorModel.getTestConnection()))
        .operations(muleConnectorModel.getOperations().stream()
            .map(this::toSerializableOperation).toList())
        .sources(muleConnectorModel.getTriggers().stream()
            .map(this::toSerializableSource).toList())
        .modelReference(muleConnectorModel.getModelReference().orElseThrow(shouldHaveRef("connector")))
        .valueProviders(muleConnectorModel.getValueProviders().stream()
            .map(this::toSerializableValueProvider).toList())
        .build();
  }

  private MuleTestConnectionSerializableModel toSerializableTestConnection(TestConnectionModel testConnectionModel) {
    if (testConnectionModel != null) {
      return MuleTestConnectionSerializableModel.builder()
          .modelReference(testConnectionModel.getModelReference().orElseThrow(shouldHaveRef("test connection")))
          .build();
    }
    return null;
  }

  private MuleConnectionProviderSerializableModel toSerializableConnectionProvider(MuleConnectionProviderModel muleConnectionProviderModel) {
    return MuleConnectionProviderSerializableModel.builder()
        .inputType(muleTypeTransformer
            .toMuleTypeWithAnnotations((WeaveType) muleConnectionProviderModel
                .getInputType().getDataType()))
        .testConnection(toSerializableTestConnection(muleConnectionProviderModel.getTestConnection()))
        .authenticationType(toSerializableAuthType(muleConnectionProviderModel.getAuthenticationType()))
        .hasExtensions(muleConnectionProviderModel.getHasExtensions())
        .modelReference(muleConnectionProviderModel.getModelReference().orElseThrow(shouldHaveRef("connection provider")))
        .name(muleConnectionProviderModel.getName())
        .build();
  }

  private MuleOperationSerializableModel toSerializableOperation(OperationModel operationModel) {
    return MuleOperationSerializableModel.builder()
        .inputType(transformOperationInputType((WeaveType) operationModel
            .getInputType().getDataType()))
        .outputType(muleTypeTransformer.toMuleTypeWithAnnotations((WeaveType) operationModel.getOutputType().getDataType()))
        .name(operationModel.getName())
        .displayName(operationModel.getDisplayName())
        .errorOutputType(muleTypeTransformer
            .toMuleTypeWithAnnotations((WeaveType) operationModel.getErrorOutputType().getDataType()))
        .isPaginated(operationModel.isPaginated())
        .errorModelList(operationModel.getErrorDefinitions().stream()
            .map(this::toSerializableErrorModel).toList())
        .inputResolvedProviders(ModelAPIToSerializedTransformer
            .transformResolvedProviders(operationModel.getInputResolvedProviders()))
        .outputResolvedProviders(ModelAPIToSerializedTransformer
            .transformResolvedProviders(operationModel.getOutputResolvedProviders()))
        .modelReference(operationModel.getModelReference().orElseThrow(shouldHaveRef("operation")))
        .build();
  }

  /**
   * Generic method to transform WeaveType to MetadataType with custom field processing.
   *
   * @param weaveType the original DataWeave type to transform
   * @param fieldProcessor function to process individual fields and determine their final type
   * @return a {@link MetadataType} suitable for serialization in the Mule model
   */
  private MetadataType transformWeaveTypeWithFieldProcessing(WeaveType weaveType,
                                                             Function<ObjectFieldType, MetadataType> fieldProcessor) {

    MetadataType baseType = muleTypeTransformer.toMuleTypeWithAnnotations(weaveType);

    if (!(baseType instanceof ObjectType objectType)) {
      return baseType;
    }

    return transformObjectTypeWithFieldProcessing(objectType, fieldProcessor);
  }

  /**
   * Generic method to transform ObjectType with custom field processing.
   *
   * @param objectType the ObjectType to transform
   * @param fieldProcessor function to process individual fields and determine their final type
   * @return a {@link MetadataType} suitable for serialization in the Mule model
   */
  private MetadataType transformObjectTypeWithFieldProcessing(ObjectType objectType,
                                                              Function<ObjectFieldType, MetadataType> fieldProcessor) {
    BaseTypeBuilder typeBuilder = BaseTypeBuilder.create(MetadataFormat.JAVA);
    ObjectTypeBuilder objectBuilder = typeBuilder.objectType();

    objectType.getAnnotations().forEach(objectBuilder::with);

    for (ObjectFieldType field : objectType.getFields()) {
      ObjectFieldTypeBuilder fieldBuilder = objectBuilder.addField();
      fieldBuilder.key(field.getKey().getName());

      MetadataType fieldType = fieldProcessor.apply(field);
      fieldBuilder.value(fieldType);

      // Preserve required, repeated, and existing annotations
      fieldBuilder.required(field.isRequired());
      fieldBuilder.repeated(field.isRepeated());
      // Copy annotations defined on the key itself (label, description, etc.)
      field.getKey().getAnnotations().forEach(fieldBuilder::withKeyAnnotation);
      field.getAnnotations().forEach(fieldBuilder::with);
    }

    return objectBuilder.build();
  }

  /**
   * Converts an error output WeaveType of valueProvider into a MetadataType, applying a special conversion for certain fields
   * (e.g., those carrying generic ErrorType (TypeParameter)).
   *
   * @param weaveType the original DataWeave type that represents the error output type
   * @return a {@link MetadataType} suitable for serialization in the Mule model
   */
  private MetadataType transformTypeParameter(WeaveType weaveType) {
    if (weaveType instanceof TypeParameter) {
      return MuleTypeTransformer.toMapMuleType();
    }
    return transformWeaveTypeWithFieldProcessing(weaveType, field -> {
      MetadataType fieldType = field.getValue();

      // Check if fieldType is ObjectType and has fields with TypeParameter
      if (fieldType instanceof ObjectType fieldObjectType) {
        fieldType = transformObjectTypeWithFieldProcessing(fieldObjectType, nestedField -> {
          MetadataType nestedFieldValue = nestedField.getValue();
          // Check if this specific field has TypeParameterType with name "ErrorType"
          if (nestedFieldValue instanceof TypeParameterType typeParameterType &&
              typeParameterType.getName().equals(GENERIC_ERROR_TYPE_FIELD)) {
            return MuleTypeTransformer.toMapMuleType();
          }
          return nestedFieldValue;
        });
      }
      return fieldType;
    });
  }

  /**
   * Converts an operation input WeaveType into a MetadataType, applying a special conversion for certain fields (e.g., those
   * carrying headers or query-params maps).
   *
   * @param weaveType the original DataWeave type that represents the operation input
   * @return a {@link MetadataType} suitable for serialization in the Mule model
   */
  private MetadataType transformOperationInputType(WeaveType weaveType) {
    return transformWeaveTypeWithFieldProcessing(weaveType, field -> {
      if (CUSTOM_INPUT_FIELDS.contains(field.getKey().getName().getLocalPart())) {
        return MuleTypeTransformer.toMapMuleType();
      } else {
        return field.getValue();
      }
    });
  }

  private MuleSourceSerializableModel toSerializableSource(TriggerModel triggerModel) {
    return MuleSourceSerializableModel.builder()
        .inputType(muleTypeTransformer.toMuleTypeWithAnnotations((WeaveType) triggerModel
            .getInputType().getDataType()))
        .outputType(muleTypeTransformer.toMuleTypeWithAnnotations((WeaveType) triggerModel.getOutputType().getDataType()))
        .name(triggerModel.getName())
        .displayName(triggerModel.getDisplayName())
        .errorOutputType(muleTypeTransformer
            .toMuleTypeWithAnnotations((WeaveType) triggerModel.getErrorOutputType().getDataType()))
        .isPaginated(triggerModel.isPaginated())
        .errorModelList(triggerModel.getErrorDefinitions().stream()
            .map(this::toSerializableErrorModel).toList())
        .inputResolvedProviders(ModelAPIToSerializedTransformer
            .transformResolvedProviders(triggerModel.getInputResolvedProviders()))
        .outputResolvedProviders(ModelAPIToSerializedTransformer
            .transformResolvedProviders(triggerModel.getOutputResolvedProviders()))
        .modelReference(triggerModel.getModelReference().orElseThrow(shouldHaveRef("trigger")))
        .build();
  }

  private MuleValueProviderSerializableModel toSerializableValueProvider(ValueProviderModel valueProviderModel) {
    return MuleValueProviderSerializableModel.builder()
        .name(valueProviderModel.getName())
        .isPaginated(valueProviderModel.isPaginated())
        .modelReference(valueProviderModel.getModelReference().orElse(null))
        .inputType(transformTypeParameter((WeaveType) valueProviderModel
            .getInputType().getDataType()))
        .providedValueType(transformTypeParameter((WeaveType) valueProviderModel
            .getProvidedValueType().getDataType()))
        .errorOutputType(transformTypeParameter((WeaveType) valueProviderModel
            .getErrorOutputType().getDataType()))
        .displayPropertiesType(transformTypeParameter((WeaveType) valueProviderModel
            .getDisplayPropertiesType().getDataType()))
        .build();
  }

  private MuleErrorSerializableModel toSerializableErrorModel(ErrorModel errorModel) {
    return MuleErrorSerializableModel.builder()
        .errorType(muleTypeTransformer.toMuleTypeWithAnnotations((WeaveType) errorModel.getErrorType().getDataType()))
        .kind(errorModel.getKind())
        .categories(errorModel.getCategories())
        .build();
  }

  private HttpAuthenticationType toSerializableAuthType(com.mulesoft.connectivity.linkweave.api.model.connection.AuthenticationType authenticationType) {
    if (authenticationType instanceof com.mulesoft.connectivity.linkweave.api.model.connection.oauth.OAuth2AuthCodeAuthenticationType oAuth2AuthCodeAuthenticationType) {
      return new OAuth2AuthCodeAuthenticationType(oAuth2AuthCodeAuthenticationType.getRefreshUrl(),
                                                  oAuth2AuthCodeAuthenticationType.getScopes(),
                                                  oAuth2AuthCodeAuthenticationType.getAuthorizationUrl(),
                                                  oAuth2AuthCodeAuthenticationType.getTokenUrl());
    } else if (authenticationType instanceof com.mulesoft.connectivity.linkweave.api.model.connection.oauth.OAuth2ClientCredentialsAuthenticationType oAuth2ClientCredentialsAuthenticationType) {
      return new OAuth2ClientCredentialsAuthenticationType(oAuth2ClientCredentialsAuthenticationType.getRefreshUrl(),
                                                           oAuth2ClientCredentialsAuthenticationType.getScopes(),
                                                           oAuth2ClientCredentialsAuthenticationType.getTokenUrl());
    } else if (authenticationType instanceof com.mulesoft.connectivity.linkweave.api.model.connection.ApiKeyHttpAuthenticationType apiKeyHttpAuthenticationType) {
      return new ApiKeyHttpAuthenticationType(apiKeyHttpAuthenticationType.getSubType().orElse(null),
                                              ApiKeyHttpAuthenticationType.In
                                                  .fromString(apiKeyHttpAuthenticationType.getIn().toString()),
                                              apiKeyHttpAuthenticationType.getName());
    } else if (authenticationType instanceof com.mulesoft.connectivity.linkweave.api.model.connection.HttpAuthenticationType httpAuthenticationType) {
      return new HttpAuthenticationType(
                                        HttpAuthenticationType.Type.fromString(httpAuthenticationType.getType().toString()),
                                        httpAuthenticationType.getSubType().orElse(null));
    }
    throw new RuntimeException("Unsupported authenticationType: " + authenticationType.getClass().getName());
  }

}
