/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.loader;

import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.CATEGORY_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.CONNECTIONS_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.DESCRIPTION_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.DISPLAY_NAME_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.NAME_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.OPERATIONS_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.TEST_CONNECTION_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.TRIGGERS_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.VALUE_PROVIDERS_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.VENDOR_PROPERTY;
import static com.mulesoft.connectivity.mule.persistence.utils.ConnectorPropertyNames.VERSION_PROPERTY;

import org.mule.runtime.api.meta.Category;
import org.mule.weave.v2.api.tooling.ts.ObjectType;

import com.mulesoft.connectivity.linkweave.api.DataWeaveObjectReference;
import com.mulesoft.connectivity.linkweave.api.DataWeaveReference;
import com.mulesoft.connectivity.linkweave.api.loader.ConnectorLoader;
import com.mulesoft.connectivity.linkweave.api.loader.ModelLoader;
import com.mulesoft.connectivity.linkweave.api.model.connection.TestConnectionModel;
import com.mulesoft.connectivity.linkweave.api.model.operation.OperationModel;
import com.mulesoft.connectivity.linkweave.api.model.provider.ValueProviderModel;
import com.mulesoft.connectivity.linkweave.api.model.trigger.TriggerModel;
import com.mulesoft.connectivity.mule.internal.model.MuleConnectionProviderModel;
import com.mulesoft.connectivity.mule.internal.model.MuleConnectorModel;

import java.util.stream.Collectors;

import org.jspecify.annotations.NullUnmarked;

/**
 * Responsible for loading a Mule Connector Model.
 */
public class MuleConnectorLoader
    extends ConnectorLoader<MuleConnectorModel, MuleConnectorModel.Builder, DataWeaveObjectReference> {

  private final ModelLoader<TestConnectionModel, TestConnectionModel.Builder, DataWeaveObjectReference> testConnectionLoader;
  private final ModelLoader<MuleConnectionProviderModel, MuleConnectionProviderModel.Builder, DataWeaveObjectReference> connectionProviderLoader;
  private final ModelLoader<OperationModel, OperationModel.Builder, DataWeaveObjectReference> operationLoader;
  private final ModelLoader<TriggerModel, TriggerModel.Builder, DataWeaveObjectReference> triggerLoader;
  private final ModelLoader<ValueProviderModel, ValueProviderModel.Builder, DataWeaveReference<?>> valueProviderLoader;


  public MuleConnectorLoader(ModelLoader<MuleConnectionProviderModel, MuleConnectionProviderModel.Builder, DataWeaveObjectReference> connectionProviderLoader,
                             ModelLoader<TestConnectionModel, TestConnectionModel.Builder, DataWeaveObjectReference> testConnectionLoader,
                             ModelLoader<OperationModel, OperationModel.Builder, DataWeaveObjectReference> operationLoader,
                             ModelLoader<TriggerModel, TriggerModel.Builder, DataWeaveObjectReference> triggerLoader,
                             ModelLoader<ValueProviderModel, ValueProviderModel.Builder, DataWeaveReference<?>> valueProviderLoader) {

    this.connectionProviderLoader = connectionProviderLoader;
    this.testConnectionLoader = testConnectionLoader;
    this.operationLoader = operationLoader;
    this.triggerLoader = triggerLoader;
    this.valueProviderLoader = valueProviderLoader;
  }

  @Override
  protected MuleConnectorModel.Builder createBuilder() {
    return new MuleConnectorModel.Builder();
  }

  @NullUnmarked
  @Override
  protected MuleConnectorModel.Builder configureBuilder(MuleConnectorModel.Builder builder,
                                                        DataWeaveObjectReference connectorObject, String name) {
    if (!(connectorObject.getType() instanceof ObjectType)) {
      throw new IllegalStateException("Invalid connector type. Expected Object but " + connectorObject.getType().getBaseType()
          + " found");
    }

    connectorObject.getProperty(NAME_PROPERTY).map(DataWeaveReference::asString).ifPresent(builder::withName);
    connectorObject.getProperty(DISPLAY_NAME_PROPERTY).map(DataWeaveReference::asString).ifPresent(builder::withDisplayName);
    connectorObject.getProperty(VERSION_PROPERTY).map(DataWeaveReference::asString).ifPresent(builder::withVersion);
    connectorObject.getProperty(DESCRIPTION_PROPERTY).map(DataWeaveReference::asString).ifPresent(builder::withDescription);
    connectorObject.getProperty(VENDOR_PROPERTY).map(DataWeaveReference::asString).ifPresent(builder::withVendor);
    connectorObject.getProperty(CATEGORY_PROPERTY).map(DataWeaveReference::asString)
        .ifPresent(category -> builder.withCategory(Category.valueOf(category.toUpperCase())));
    builder.withModelReference(connectorObject.getPath());

    // Connections
    builder.withConnections(connectorObject.requireObjectProperty(CONNECTIONS_PROPERTY).properties()
        .map(e -> connectionProviderLoader.loadModel(e.getValue().asObjectReference(), e.getKey()))
        .collect(Collectors.toUnmodifiableList()));

    // Test Connection
    connectorObject.getObjectProperty(TEST_CONNECTION_PROPERTY)
        .ifPresent(tc -> builder.withTestConnection(testConnectionLoader.loadModel(tc, name)));

    // Operations
    builder.withOperations(connectorObject.requireObjectProperty(OPERATIONS_PROPERTY).properties()
        .map(e -> operationLoader.loadModel(e.getValue().asObjectReference(), null))
        .toList());

    // Triggers
    builder.withTriggers(connectorObject.getObjectProperty(TRIGGERS_PROPERTY)
        .stream()
        .flatMap(DataWeaveObjectReference::properties)
        .map(e -> triggerLoader.loadModel(e.getValue().asObjectReference(), e.getKey()))
        .toList());

    // Value Providers
    connectorObject.getObjectProperty(VALUE_PROVIDERS_PROPERTY).ifPresent(vp -> builder.withValueProviders(vp.properties()
        .map(e -> valueProviderLoader.loadModel(e.getValue(), e.getKey())).collect(Collectors.toList())));

    return builder;
  }
}
