/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.metadata;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.SemanticTermsTypeAnnotation;
import org.mule.weave.v2.api.tooling.ts.ArrayMetadataValue;
import org.mule.weave.v2.api.tooling.ts.DWMetadataValue;
import org.mule.weave.v2.ts.MetadataValue;

import com.mulesoft.connectivity.linkweave.api.loader.InvalidConnectorException;
import com.mulesoft.connectivity.linkweave.api.metadata.MetadataValueUtils;

import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Responsible for creating {@link SemanticTermsTypeAnnotation} instances from {@link MetadataValue}.
 */
public class SemanticTermsTypeAnnotationFactory implements TypeAnnotationFactory {

  // Same as org.mule.runtime.extension.api.declaration.type.annotation.SemanticTermsTypeAnnotation.getName
  public static final String SEMANTIC_TERMS_ANNOTATION_KEY = "semanticTerms";
  public static final String ANNOTATION_DESCRIPTION = "The 'semanticTerms' metadata";

  @Override
  public String getKey() {
    return SEMANTIC_TERMS_ANNOTATION_KEY;
  }

  @Override
  public TypeAnnotation create(MetadataValue metadataValue) {
    return new SemanticTermsTypeAnnotation(asSet(metadataValue, ANNOTATION_DESCRIPTION));
  }

  /**
   * Returns a metadata value representing an array as a set of sorted strings.
   *
   * @param metadataValue the metadata value
   * @param what a description for errors
   * @return the set of strings
   * @throws InvalidConnectorException if metadataValue does not represent an array or not all its elements are strings
   */
  public static Set<String> asSet(DWMetadataValue metadataValue, String what) {
    if (!(metadataValue instanceof ArrayMetadataValue)) {
      throw new InvalidConnectorException(what + " was supposed to be an array, but it's "
          + MetadataValueUtils.describeType(metadataValue) + " instead.", metadataValue.getLocation());
    }
    return Stream.of(((ArrayMetadataValue) metadataValue).getElements())
        .map(mv -> MetadataValueUtils.asString(mv, what + " element"))
        .collect(Collectors.toCollection(TreeSet::new));
  }
}
