/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.persistence;

import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;

import static java.util.Optional.empty;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.api.model.IntersectionType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import com.mulesoft.connectivity.mule.persistence.serialize.ReferenceResolver;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * {@link MetadataTypeVisitor} implementation that registers all the {@link ObjectType}s declared inside another
 * {@link MetadataType} if any to create a type catalog.
 */
public class CatalogTypeCollector extends MetadataTypeVisitor implements ReferenceResolver {

  private final LinkedHashMap<String, ObjectType> registeredTypes = new LinkedHashMap<>();
  private final List<String> index = new ArrayList<>();

  @Override
  public void visitObject(ObjectType type) {
    String id = createIdentifier(type);
    if (registeredTypes.containsKey(id)) {
      return;
    }
    registeredTypes.put(id, type);
    index.add(id);

    type.getOpenRestriction().ifPresent(t -> t.accept(this));
    type.getFields().forEach(field -> field.getValue().accept(this));
  }

  @Override
  public void visitArrayType(ArrayType arrayType) {
    arrayType.getType().accept(this);
  }

  @Override
  public void visitFunction(FunctionType functionType) {
    // Not relevant in the context of MetadataTypes
    throw new UnsupportedOperationException("FunctionType is not supported in MetadataType");
  }

  @Override
  public void visitIntersection(IntersectionType intersectionType) {
    intersectionType.getTypes().forEach(t -> t.accept(this));
  }

  @Override
  public void visitUnion(UnionType unionType) {
    unionType.getTypes().forEach(t -> t.accept(this));
  }

  public Map<String, ObjectType> getCatalog() {
    Map<String, ObjectType> copy = new LinkedHashMap<>(registeredTypes.size());

    List<String> keys = new ArrayList<>(registeredTypes.keySet());
    for (String key : keys) {
      copy.put(referenceIndex(key), registeredTypes.get(key));
    }

    return copy;
  }

  private String referenceIndex(String key) {
    return "0x" + Integer.toHexString(index.indexOf(key) + 1);
  }

  @Override
  public Optional<String> getIdentifier(ObjectType type) {
    String id = createIdentifier(type);
    return registeredTypes.containsKey(id) ? Optional.of(referenceIndex(id)) : empty();
  }

  private String createIdentifier(ObjectType type) {
    Optional<String> typeId = getTypeId(type);
    return typeId.orElseGet(() -> String.valueOf(System.identityHashCode(type)));
  }
}
