/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.persistence;

import static org.mule.metadata.persistence.MetadataTypeConstants.REF_FLAG;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE;

import static java.util.Optional.empty;

import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.MetadataSerializingException;
import org.mule.metadata.persistence.ObjectTypeReferenceHandler;

import com.mulesoft.connectivity.mule.persistence.serialize.ReferenceResolver;

import java.io.IOException;
import java.util.Objects;
import java.util.Optional;

import com.google.gson.stream.JsonWriter;

/**
 * {@link ObjectTypeReferenceHandler} implementation for type serialization that declare an external type catalog in its
 * serialized structure.
 */
public class ExternalCatalogObjectTypeReferenceHandler implements ObjectTypeReferenceHandler {

  private final ReferenceResolver referenceResolver;

  public ExternalCatalogObjectTypeReferenceHandler(ReferenceResolver referenceResolver) {
    this.referenceResolver = referenceResolver;
  }

  /**
   * We don't need to read the reference catalog object here as this will only take part in the serialization process.
   *
   * @param typeReference a type id
   * @return empty Optional object
   */
  @Override
  public Optional<TypeBuilder> readReference(String typeReference) {
    return empty();
  }

  /**
   * Writes a type reference if this has an associated type id or uses the identity hashcode to be later fetched from the catalog.
   */
  @Override
  public Optional<String> writeReference(ObjectType type, JsonWriter writer) {
    Objects.requireNonNull(referenceResolver, "Tried to write a reference on a read-only handler");
    Optional<String> referenceOptional = referenceResolver.getIdentifier(type);
    if (referenceOptional.isPresent()) {
      String reference = referenceOptional.get();
      try {
        writer.name(TYPE).value(REF_FLAG + reference);
      } catch (IOException e) {
        throw new MetadataSerializingException("MetadataType reference [" + reference + "]", e);
      }
    }
    return referenceOptional;
  }
}
