/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.connectivity.mule.internal.utils;

import static java.util.function.Function.identity;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.weave.v2.metadata.api.MetadataModelConverter;
import org.mule.weave.v2.metadata.api.MetadataModelConverterConfig;
import org.mule.weave.v2.metadata.api.WeaveTypeMetadataConvert;
import org.mule.weave.v2.ts.WeaveType;

import com.mulesoft.connectivity.mule.internal.metadata.DefaultValueAnnotationFactory;
import com.mulesoft.connectivity.mule.internal.metadata.DescriptionTypeAnnotationFactory;
import com.mulesoft.connectivity.mule.internal.metadata.FormatAnnotationFactory;
import com.mulesoft.connectivity.mule.internal.metadata.LabelTypeAnnotationFactory;
import com.mulesoft.connectivity.mule.internal.metadata.SemanticTermsTypeAnnotationFactory;
import com.mulesoft.connectivity.mule.internal.metadata.TypeAnnotationFactory;

import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import scala.Option;
import scala.Predef;
import scala.collection.JavaConverters;

/**
 * Responsible for converting Weave type to Mule MetadataType.
 */
public class MuleTypeTransformer {

  private final MetadataModelConverter metadataModelConverter;

  public MuleTypeTransformer() {
    Map<String, TypeAnnotationFactory> annotationFactories = getAnnotationFactories();

    WeaveTypeMetadataConvert weaveTypeMetadataConvert = metadata -> {
      TypeAnnotationFactory factory = annotationFactories.get(metadata.name());
      return factory != null ? factory.create(metadata.value()) : null;
    };

    var config = MetadataModelConverterConfig.apply(false,
                                                    false,
                                                    true,
                                                    true,
                                                    false,
                                                    Option.apply(weaveTypeMetadataConvert));

    metadataModelConverter = new MetadataModelConverter(config);
  }

  private Map<String, TypeAnnotationFactory> getAnnotationFactories() {
    return Stream.of(
                     new DescriptionTypeAnnotationFactory(),
                     new LabelTypeAnnotationFactory(),
                     new FormatAnnotationFactory(),
                     new DefaultValueAnnotationFactory(),
                     new SemanticTermsTypeAnnotationFactory())
        .collect(Collectors.toMap(TypeAnnotationFactory::getKey, identity()));
  }

  /**
   * Converts a {@link WeaveType} into a {@link MetadataType} with default annotations.
   * <p>
   * This method uses a predefined set of annotation factories to annotate the resulting {@link MetadataType}.
   *
   * @param inputType The type to convert
   * @return An equivalent type represented using the {@code mule-metadata-api} library, with annotations applied
   */
  public MetadataType toMuleTypeWithAnnotations(WeaveType inputType) {
    scala.collection.immutable.Map<String, String> scalaMimeTypeProperties =
        JavaConverters.mapAsScalaMap(Map.<String, String>of()).toMap(Predef.conforms());
    return metadataModelConverter.toMuleType(inputType, MetadataFormat.JAVA, Option.empty(), scalaMimeTypeProperties);
  }

  /**
   * Creates a Map type for custom headers and custom query parameters.
   *
   * @return a MetadataType representing a Map with String keys and String values
   */
  public static MetadataType toMapMuleType() {
    BaseTypeBuilder typeBuilder = BaseTypeBuilder.create(MetadataFormat.JAVA);
    return typeBuilder.objectType()
        .openWith(typeBuilder.stringType())
        .build();
  }
}
