/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.7
import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Annotations
import * from com::mulesoft::connectivity::decorator::Operation
import * from com::mulesoft::connectivity::decorator::PaginationStrategies
import * from com::mulesoft::connectivity::decorator::Trigger
import * from com::mulesoft::connectivity::Metadata
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::decorator::ValueProvider
import * from com::mulesoft::connectivity::decorator::Executor
import * from com::mulesoft::connectivity::metadata::Builder
import * from com::mulesoft::connectivity::Types

// ----------------
// ---- Types -----
// ----------------

type Types = {
  Pet: Types.NewPet & { id: Number, createdAt?: LWDate },
  Error: {
    code: Number,
    message: String
  },
  NewPet: {
    name: @Label(value = "Name of the Pet") @HelpText(value = "The name has to go in lowercase") String,
    category: @ValuesFrom(value = {name: "categoriesValueProvider"}) Number,
    category2?: @ValuesFrom(value = {name: "categories2ValueProvider"}) Integer,
    category3?: @ValuesFrom(value = {name: "categories3ValueProvider"}) Int64,
    tag: @ValuesFrom(value = { name: "tagsValueProvider", arguments: {id: "category" } }) String,
    petType?: "dog" | "cat" | "bird" | "other",
    ranking?: 1 | 2 | 3 | 4 | 5,
    user_id?: Number,
    extraInfo?: {
      more_details?: String
    }
  },
  Tag: {
    id: String,
    label: String
  },
  Category: {
    id: Number,
    label: String
  },
  Category2: {
    id: Integer,
    label: String
  },
  Category3: {
    id: Int64,
    label: String
  },
  Field: {
    name: String,
    "type": String,
    fields?: Array<Types.Field>
  },
  User: {
    id: Number,
    info: {
      name: String,
      pets: Array<Types.Pet>,
      userTypes: Array<String>
    }
  },
  NewUser: {
    id: Number,
    info: {
      name: String,
      extraData: {
        userTypes: {
          values: Array<String>
        }
      }
    }
  },
  Ticket: {
    id: Number,
    title: String,
    extraFields: @MetadataProvider(value = { name: "ticketMetadataProvider" }) Object
  },
  Product: Types.NewProduct & { id: Number },
  NewProduct: {
    name: String,
    productType: String,
    productGroup?: @ValuesFrom(value = {name: "productGroupValueProvider"}) String,
    productTag: @ValuesFrom(value = { name: "productTagValueProvider", arguments: {id: "/extraInfo/productFamily" } }) String, // 3
    extraInfo: {
      productSubType: @ValuesFrom(value = { name: "productSubTypeValueProvider", arguments: {id: "/productType" } }) String, // 1
      productCategory: @ValuesFrom(value = { name: "productCategoryValueProvider", arguments: {id: "/extraInfo/productFamily" } }) String, // 2
      productFamily: String
    }
  },
  Client: {
    id: Number,
    extraFields: @MetadataProvider(value = { name: "clientMetadataProvider" }) Object
  },
  QueryBuilderObject: {
    id: String,
    label: String
  },
  AllTypes: {
    aString: String,
    aNumber: Number,
    anInteger: Integer,
    aFloat: Float,
    aDouble: Double,
    anInt32: Int32,
    anInt64: Int64,
    aBoolean: Boolean,
    aDate: LWDate,
    aDatetime: LWDateTime,
    aReference: Types.AReference,
    anObject: {
        anInnerField: String
    },
    anArray: Array<String>
  },
  AReference: {
    aField: String
  },
  Attachment: {
     id: String,
     filename: String,
     size: Number,
     content: String,
     mimeType?: String,
     created?: Number
   }
}

// ----------------
// -- Connection --
// ----------------

var mockConnectionProvider : HttpConnectionProvider<Object, { "type": "custom" }> = {
    authenticationType: {"type": "custom"},
    connect: (parameter) -> (httpRequest: HttpRequester) -> do {
      var path = httpRequest.path
      var response = {contentType: "application/json", status: 200, headers: {}, cookies: {}}
      var pet1 = {id: 1, name: "Name1", category: 1, tag: "Tag1", createdAt: "2024-01-01"}
      var pet2 = {id: 2, name: "Name2", category: 2, tag: "Tag2", createdAt: "2024-01-01", ranking: 5}
      var categories = if(isEmpty(parameter.categories)) [ {id: 1, label: "Cat1"}, {id: 2, label: "Cat2"}] else parameter.categories
      var tags = [ {id: "1", label: "Tag1"}, {id: "2", label: "Tag2"}]
      var accountFields = [{name: "id", "type" : "Number"},
                           {name : "description", "type": "String"},
                           {name : "extra_info", "type": "String"},
                           {name : "address", "type": "Object", fields: [{name: "street", "type": "String"}, {name: "city", "type": "String"}]},
                           {name : "userTypes", "type": "Array", of: ["String"]},
                           {name : "active", "type": "Boolean"},
                           {name : "billingStreet", "type": "Union", of: ["String", "Null"]},
                           {name : "createdAt", "type": "Date"},]
      var accounts = [{id: 1, description: "order 1", extra_info: "info 1", address: {street: "123 Main St", city: "Anytown"}, userTypes: ["type1", "type2"], active: true, billingStreet: "null", createdAt: "2024-01-01"},
                      {id: 2, description: "order 2", extra_info: "info 2", address: {street: "456 Main St", city: "Anytown"}, userTypes: ["type3", "type4"], active: false, billingStreet: "123 Main St", createdAt: "2024-01-01" }]

      var user1 = {id: 1, info: {name: "john", pets: [pet1, pet2], userTypes: ["type1", "type2"]}}

      var ticketFields = [{name: "priority", "type" : "Number"}, {name : "description", "type": "String"}]
      var tickets = [{id: 1, title: "Ticket1", extraFields: {priority: 1, description: "Desc1"}}, {id: 2, title: "Ticket2", extraFields: {priority: 2, description: "Desc2"}}]

      var productTags = [ {id: "1", label: "Tag1"}, {id: "2", label: "Tag2"}]
      var productSubTypes = [ {id: "1", label: "SubType1"}, {id: "2", label: "SubType2"}]
      var productCategories = [ {id: "1", label: "Cat1"}, {id: "2", label: "Cat2"}]
      var productTypes = [ {id: "1", label: "Type1"}, {id: "2", label: "Type2"}]
      var product = {id: 1, name: "ProductName", productType: "type1", productTag: "tag1", extraInfo: {productSubType: "subType1", productCategory: "cat1", productFamily: "family1"}}

      var clientFields = [{name: "name", "type" : "String"}, {name : "details", "type": "Object", fields: [{name: "description", "type" : "String"}]}]
      var clients = [{id: 1, name: "Name1", details: {description: "description 1"}}, {id: 2, name: "Name2", details: {description: "description 2"}}]

      var groups1 = [ {id: "1", label: "Group1"}, {id: "2", label: "Group2"}]
      var groups2 = [ {id: "3", label: "Group3"}, {id: "4", label: "Group4"}]

      var queryBuilderObjects = [ {id: "Accounts", label: "Accounts"}, {id: "Users", label: "Users"}]
      var queryBuilderAccountFields = [{name : "name", "type": "String"}, {name : "description", "type": "String"}, {name : "createdAt", "type": "DateTime"}]
      var queryBuilderUserFields = [{name : "firstName", "type": "String"}, {name : "lastName", "type": "String"}, {name : "createdAt", "type": "DateTime"}]
      var queryAccounts = [{id: 1, name: "name1", description: "desc1", createdAt: |2025-01-01T12:00:00.000-08:00|}, {id: 2, name: "name2", description: "desc2", createdAt: |2025-02-01T12:00:00.000-08:00|}]
      var queryAccounts2 = [{id: 3, name: "name3", description: "desc3", createdAt: |2025-03-01T12:00:00.000-08:00|}, {id: 4, name: "name4", description: "desc4", createdAt: |2025-04-01T12:00:00.000-08:00|}]
      var queryUsers = [{id: 1, firstName: "fn1", lastName: "ln1", createdAt: |2025-01-01T12:00:00.000-08:00|}, {id: 2, firstName: "fn2", lastName: "ln2", createdAt: |2025-02-01T12:00:00.000-08:00|}]
      var queryUsers2 = [{id: 3, firstName: "fn3", lastName: "ln3", createdAt: |2025-03-01T12:00:00.000-08:00|}, {id: 4, firstName: "fn4", lastName: "ln4", createdAt: |2025-04-01T12:00:00.000-08:00|}]

      var allTypes = {aString: "string", aNumber: 1, anInteger: 2, aFloat: 3.0, aDouble: 4.0, anInt32: 5, anInt64: 6, aBoolean: true, aDate: "2024-01-01", aDatetime: "2024-01-01T00:00:00Z", aReference: {aField: "field"}, anObject: {anInnerField: "innerField"}, anArray: ["item1", "item2"] }

      ---
      path match {
        case "/pets/1" -> response ++ {body: pet1}
        case "/users/1" -> response ++ {body: user1}
        case "/users" -> response ++ {body: user1}
        case "/categories" -> response ++ {body: categories}
        case "/categories2" -> response ++ {body: categories}
        case "/categories3" -> response ++ {body: categories}
        case "/categories/1/tags" -> response ++ {body: tags}
        case "/pets" -> response ++ {body: pet1}
        case "/pets/offset" -> do {
          var petPage = if(isEmpty(parameter.pets))
                                        if ((httpRequest.queryParams.from default 0) == 0)
                                            [pet1]
                                        else if ((httpRequest.queryParams.from default 0) == 1)
                                            [pet2]
                                        else
                                            []
                                    else
                                        if ((httpRequest.queryParams.from default 0) == 0)
                                            parameter.pets
                                        else
                                            []
          ---
          response ++ {body: petPage}
        }
        case "/pets/page-number" -> do {
          var petPage = if ((httpRequest.queryParams.page default 0) == 0)
                            [pet1]
                        else if ((httpRequest.queryParams.page default 0) == 1)
                            [pet2]
                        else
                            []
          ---
          response ++ {body: petPage}
        }
        case "/pets/marker" -> do {
          var petPage = if ((httpRequest.queryParams.markerToken default "") == "")
                            {items: [pet1], nextToken: "asd"}
                        else if ((httpRequest.queryParams.markerToken default "") == "asd")
                            {items: [pet2]}
                        else
                            []
          ---
          response ++ {body: petPage}
        }
        case "/pets/hypermedia" -> do {
          var petPage = if ((httpRequest.queryParams.from default 0) == 0)
                            {items: [pet1], nextUrl: "/pets/hypermedia?from=1"}
                        else if ((httpRequest.queryParams.from default 0) == 1)
                            {items: [pet2]}
                        else
                            []
          ---
          response ++ {body: petPage}
        }
        case "/account-metadata" -> response ++ {body: accountFields}
        case "/account" -> response ++ {body: accounts[0]}
        case "/client-metadata" -> response ++ {body: clientFields}
        case "/clients" -> response ++ {body: clients}
        case "/order-custom-fields" -> response ++ {body : {id : 1, description: "", category: "1", customField: 1 }}
        case "/ticket-metadata" -> response ++ {body: ticketFields}
        case "/tickets" -> response ++ {body: tickets}

        case "/products/family/1/tags" -> response ++ {body: productTags}
        case "/products/types/1/subTypes" -> response ++ {body: productSubTypes}
        case "/products/family/1/categories" -> response ++ {body: productCategories}
        case "/products/types" -> response ++ {body: productTypes}
        case "/products" -> do {
          var result = if (httpRequest.method == "POST")
                            product
                        else
                            [product]
          ---
          response ++ {body: result}
        }
        case "/products/groups" -> do {
          var groupsPage = if ((httpRequest.queryParams.page default 0) == 0)
                            groups1
                        else if ((httpRequest.queryParams.page default 0) == 1)
                            groups2
                        else
                            []
          ---
          response ++ {body: groupsPage}
        }
        case "/queryBuilder-objects" -> response ++ {body: queryBuilderObjects}
        case "/queryBuilder-metadata" -> do {
           var fields = if ((httpRequest.queryParams.objectType) == "Users")
                             {body: queryBuilderUserFields}
                         else
                             {body: queryBuilderAccountFields}
           ---
           response ++ fields
         }
        case "/query" -> do {
            var q = log(httpRequest.queryParams.query)
            var result = if (q == "SELECT fields(ALL) FROM Accounts")
                        queryAccounts
                    else if (q == "SELECT name, description FROM Accounts")
                        queryAccounts map ((item) -> {name: item.name, description: item.description})
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE id = 1")
                        queryAccounts filter ((item) -> item.id == 1)
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE id != null AND name = 'name1' AND description = 'desc1'")
                        queryAccounts filter ((item) -> item.name == 'name1' and item.description == 'desc1' )
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE id = 1 OR id = 2")
                        queryAccounts filter ((item) -> item.id == 1 or item.id == 2 )
                    else if (q == "SELECT fields(ALL) FROM Accounts")
                        queryAccounts orderBy ((item) -> -item.id )
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE id = 2 AND (name = 'name1' OR name = 'name2')")
                        queryAccounts filter ((item) -> item.id == 2 and (item.name == 'name1' or item.name == 'name2') )
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE name in ('name1', 'name2', 'name3')")
                        queryAccounts
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE id between 2 and 5")
                        queryAccounts filter ((item) -> item.id >= 2)
                    else if (q == "SELECT fields(ALL) FROM Users")
                        queryUsers
                    else if (q == "SELECT fields(ALL) FROM Users OFFSET 2")
                        queryUsers2
                    else if (q == "SELECT fields(ALL) FROM Accounts")
                         queryAccounts
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE createdAt>=2025-01-01T12:00:00Z OFFSET 2")
                         queryAccounts2
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE createdAt>=2025-01-01T12:00:00Z")
                        queryAccounts
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE createdAt>=2025-02-01T12:00:00-08:00")
                        queryAccounts2
                    else if (q == "SELECT fields(ALL) FROM Accounts ORDER BY id DESC, name ASC")
                        queryAccounts orderBy -$$
                    else if (q == "SELECT fields(ALL) FROM Accounts WHERE name startsWith 'na' AND description contains '1'")
                        queryAccounts filter ((item) -> (item.name startsWith 'na') and (item.description contains '1'))
                    else if (q == "SELECT fields(ALL) FROM Accounts LIMIT 1")
                         [queryAccounts[0]]
                    else
                        []
            ---
            response ++ {body: result}
        }
        case "/all-types" -> response ++ {body: allTypes}
        case "/rest/api/3/attachment/content/10001" -> response update {
          case .headers."Content-Disposition"! -> "attachment; filename*=UTF-8''picture.jpg"
          case body at .body! -> ("binary-content-picture" as Binary {encoding: "UTF-8"})
        }
        case "/rest/api/3/item/TEST-123/attachments" -> do {
          var size = sizeOf((httpRequest.body default ("") ) as Binary)
          var body =
            {
              content: "https://api.example.com/rest/api/3/attachment/content/10001",
              filename: "picture.jpg",
              id: "10001",
              mimeType: "image/jpeg",
              self: "https://api.example.com/rest/api/3/attachments/10001",
              size: size,
            }
          ---
          response ++ {body: body}
        }
        else -> response
      }
    }
}

var connections = {
  basicAuth: defineBasicHttpConnectionProvider<BasicAuthSchema & { baseUri: String }>(
                 (connectionInfo) -> {username: connectionInfo.username, password: connectionInfo.password},
                 (parameter) -> {baseUri: parameter.baseUri}
             ),
  oAuth: defineOAuth2Connection<{
       baseUri: @HelpText(value = "add the baseUri with http protocol") String,
       accessToken: @SemanticTerms(value = ["password"]) String
  }>(
       (parameter) -> {accessToken: parameter.accessToken},
       (parameter) -> {baseUri: parameter.baseUri},
       {
           grantType: 'authorizationCode',
           authorizationUrl: "http://localhost/authorize",
           tokenUrl: "http://localhost/token",
           scopes: ["test"]
       }
     ),
  oAuthClientCredentials: defineOAuth2Connection<{
       baseUri: @HelpText(value = "add the baseUri with http protocol") String,
       accessToken: @SemanticTerms(value = ["password"]) String
  }>(
       (parameter) -> {accessToken: parameter.accessToken},
       (parameter) -> {baseUri: parameter.baseUri},
       {
           grantType: 'clientCredentials',
           refreshUrl: "http://localhost/refresh",
           tokenUrl: "http://localhost/token",
           scopes: ["test"]
       }
     ),
  apiKeyAuth: defineApiKeyHttpConnectionProvider<ApiKeyAuthSchema & {
      baseUri: @HelpText(value = "add the baseUri with http protocol") String
  }>(
       (schema) -> {apiKey: schema.apiKey},
       (schema) -> {baseUri: schema.baseUri},
       {in: 'header', name: "X-API-KEY"}
  ),
  bearerAuth: defineBearerHttpConnectionProvider<BearerAuthSchema & {
      baseUri: @HelpText(value = "add the baseUri with http protocol") String
  }>(
       (schema) -> {token: schema.token},
       (schema) -> {baseUri: schema.baseUri}
  ),
  mockAuth: mockConnectionProvider,
  basicAuthWithExtensions: defineBasicHttpConnectionProvider<BasicAuthSchema & { baseUri: String } & { extraKey: String }>(
                 (connectionInfo) -> {username: connectionInfo.username, password: connectionInfo.password},
                 (parameter) -> {baseUri: parameter.baseUri},
                 (parameter) -> [
                                    {in: "header", name:"customHeaderDynamic", value: parameter.extraKey},
                                    {in: "query", name:"customQueryStatic", value: "myValue"}
                                ]
             ),
  oAuthClientCredentialsWithExtensions: defineOAuth2Connection<OAuth2AuthSchema & {baseUri: String} & { tenantid: String }>(
       (parameter) -> {accessToken: parameter.accessToken},
       (parameter) -> {baseUri: parameter.baseUri},
       {
           grantType: 'clientCredentials',
           refreshUrl: "http://localhost/refresh",
           tokenUrl: "http://localhost/token",
           scopes: ["test"]
       },
       (parameter) ->   [
                            {inDance: false, in: "header", name:"tenantid", value: parameter.tenantid},
                            {inDance: true, in: "header", name:"customHeaderStatic", value: "1234"}
                        ]
     ),
     awsAuth: defineAwsHttpConnectionProvider<AwsAuthSchema & {
           baseUri: @HelpText(value = "add the baseUri with http protocol") String,
           serviceName: @Template(value = "sns") @Hidden() String
       }>(
            (schema) -> {accessKey: schema.accessKey, secretKey: schema.secretKey, region: schema.region, useSessionToken: schema.useSessionToken, serviceName: schema.serviceName },
            (schema) -> {baseUri: schema.baseUri}
       )
}


// ----------------
// -- Operations --
// ----------------

type OperationTypes = {
  "/users/{id}": {
    get: {
      response200: Types.User,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        uri: {|
          id: Number
        |},
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Types.User | Types.Error
    }
  },
  "/users": {
    post: {
      response200: Types.User,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object,
        body: Types.NewUser
      |}>,
      response: Types.User | Types.Error
    }
  },
  "/pets/{id}": {
    get: {
      response200: Types.Pet,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        uri: {|
          id: Number
        |},
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Types.Pet | Types.Error
    }
  },
  "/pets": {
    post: {
      response200: Types.Pet,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object,
        body: Types.NewPet
      |}>,
      response: Types.Pet | Types.Error
    }
  },
  "/pets/offset": {
      get: {
        response200: Array<Types.Pet>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: {
            category?: Number,
            limit?: Number,
            from?: Number
          },
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Pet> | Types.Error
      }
    },
  "/pets/page-number": {
      get: {
        response200: Array<Types.Pet>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: {
            category?: Number,
            limit?: Number,
            page?: Number
          },
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Pet> | Types.Error
      }
    },
  "/pets/marker": {
      get: {
        response200: {items: Array<Types.Pet>, nextToken?: String},
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: {
            category?: Number,
            markerToken?: String
          },
          headers: Object,
          cookie: Object
        |}>,
        response: {items: Array<Types.Pet>, nextToken?: String} | Types.Error
      }
    },
  "/pets/hypermedia": {
      get: {
        response200: {items: Array<Types.Pet>, nextUrl?: String},
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: {
            category?: Number,
            limit?: Number,
            from?: Number
          },
          headers: Object,
          cookie: Object
        |}>,
        response: {items: Array<Types.Pet>, nextUrl?: String} | Types.Error
      }
    },
  "/categories": {
    get: {
      response200: Array<Types.Category>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Category> | Types.Error
    }
  },
  "/categories2": {
    get: {
      response200: Array<Types.Category2>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Category2> | Types.Error
    }
  },
  "/categories3": {
    get: {
      response200: Array<Types.Category3>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Category3> | Types.Error
    }
  },
  "/categories/{id}/tags": {
      get: {
        response200: Array<Types.Tag>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          uri: {|
            id: Number
          |},
          query: Object,
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Tag> | Types.Error
      }
  },
  "/account": {
      post: {
        response200: @MetadataProvider(value = { name: "accountMetadataProvider" }) Object,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object,
          body: @MetadataProvider(value = { name: "accountMetadataProvider" }) Object
        |}>,
        response: Object | Types.Error
      }
  },
  "/record": {
      post: {
        response200: @MetadataProvider(value = { name: "recordMetadataProvider", arguments: { recordType: "/recordType" } }) Object,
        responsedefault: Types.Error,
        request: {
            recordType: String,
            record: @MetadataProvider(value = { name: "recordMetadataProvider", arguments: { recordType: "/recordType" } }) Object
        },
        response: Object | Types.Error
      }
  },
  "/record-nested-medatadata-provider-param": {
      post: {
        response200: @MetadataProvider(value = { name: "recordMetadataProvider", arguments: { recordType: "/info/recordType" } }) Object,
        responsedefault: Types.Error,
        request: {
            info: {"recordType": String},
            record: @MetadataProvider(value = { name: "recordMetadataProvider", arguments: { recordType: "/info/recordType" } }) Object
        },
        response: Object | Types.Error
      }
  },
  "/account-metadata": {
      get: {
        response200 : Array<Types.Field>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Field> | Types.Error
      }
  },
  "/tickets": {
      get: {
        response200: Array<Types.Ticket>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object,
        |}>,
        response: Array<Types.Ticket> | Types.Error
      }
  },
  "/ticket-metadata": {
      get: {
        response200 : Array<Types.Field>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Field> | Types.Error
      }
  },
  "/order-custom-fields" : {
      post: {
        response200: @MetadataProvider(value = { name: "orderMetadataProvider" }) Order,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object,
          body: @MetadataProvider(value = { name: "orderMetadataProvider" }) Order
        |}>,
        response: Order | Types.Error
      }
  },
  "/products/family/{id}/tags": {
    get: {
      response200: Array<Types.Tag>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        uri: {|
          id: String
        |},
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Tag> | Types.Error
    }
  },
  "/products/types/{id}/subTypes": {
    get: {
      response200: Array<Types.Tag>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        uri: {|
          id: String
        |},
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Tag> | Types.Error
    }
  },
  "/products/family/{id}/categories": {
    get: {
      response200: Array<Types.Tag>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        uri: {|
          id: String
        |},
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Tag> | Types.Error
    }
  },
  "/products": {
    post: {
      response200: Types.Product,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object,
        body: Types.NewProduct
      |}>,
      response: Types.Product | Types.Error
    },
    get: {
      response200: Array<Types.Product>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: {
          productType?: String
        },
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Product> | Types.Error
    }
  },
  "/products/types": {
    get: {
      response200: Array<Types.Tag>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.Tag> | Types.Error
    }
  },
  "/clients": {
      get: {
        response200: Array<Types.Client>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Client> | Types.Error
      }
  },
  "/client-metadata": {
      get: {
        response200 : Array<Types.Field>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Field> | Types.Error
      }
  },
  "/products/groups": {
      get: {
        response200: Array<Types.Tag>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: {
            page?: Number,
            limit?: Number,
          },
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Tag> | Types.Error
      }
    },
  "/query": {
      get: {
        response200: Array<Object>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: {
            query: String
          },
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Object> | Types.Error
      }
    },
  "/queryBuilder-objects": {
    get: {
      response200: Array<Types.QueryBuilderObject>,
      responsedefault: Types.Error,
      request: HttpRequestType<{|
        query: Object,
        headers: Object,
        cookie: Object
      |}>,
      response: Array<Types.QueryBuilderObject> | Types.Error
    }
  },
  "/queryBuilder-metadata": {
      get: {
        response200 : Array<Types.Field>,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: {
            objectType: String
          },
          headers: Object,
          cookie: Object
        |}>,
        response: Array<Types.Field> | Types.Error
      }
  },
  "/all-types": {
    post: {
        response200: Types.AllTypes,
        responsedefault: Types.Error,
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object,
          body: Types.AllTypes
        |}>,
        response: Types.AllTypes | Types.Error
      }
  },
  "/rest/api/3/attachment/content/{id}": {
      get: {
        request: HttpRequestType<{|
          uri: {|
            id: String
          |},
          query: { redirect?: Boolean },
          headers: Object,
          cookie: Object
        |}>,
        response200: Binary,
        response: Binary
      }
    },
  "/rest/api/3/item/{resourceId}/attachments": {
      post: {
        request: HttpRequestType<{|
          uri: {|
            resourceId: String
          |},
          query: Object,
          headers: Object,
          cookie: Object,
          body: Binary
        |}>,
        response200: Types.Attachment,
        response: Types.Attachment
      }
    },
    "/newstyle-errors": {
      get: {
        request: HttpRequestType<{|
          query: Object,
          headers: Object,
          cookie: Object
        |}>,
        response200: { successMessage: String },
        responsedefault: { errorMessage: String }
      }
    }
  }


var operations = {
  "/users/{id}": {
    get: {
      name: "findUserById",
      displayName: "find user by id",
      executor: (parameter: OperationTypes."/users/{id}".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/users/{id}".get.response200>, ResultFailure<HttpResponse<OperationTypes."/users/{id}".get.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "GET",
            path: "/users/$(uri.id)",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            config: {
                contentType: "application/json"
            },
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/users/{id}".get.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/users/{id}".get.responsedefault>)
          }
        }
    }
  },
  "/users": {
    post: {
        name: "addUser",
        displayName: "addUser",
        executor: (parameter: OperationTypes."/users".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/users".post.response200>, ResultFailure<HttpResponse<OperationTypes."/users".post.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "POST",
              path: "/users",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/users".post.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/users".post.responsedefault>)
            }
          }
      }
  },
  "/pets/{id}": {
    get: {
      name: "findPetById",
      displayName: "find pet by id",
      executor: (parameter: OperationTypes."/pets/{id}".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/pets/{id}".get.response200>, ResultFailure<HttpResponse<OperationTypes."/pets/{id}".get.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "GET",
            path: "/pets/$(uri.id)",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/pets/{id}".get.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/pets/{id}".get.responsedefault>)
          }
        }
    }
  },
  "/pets": {
    post: {
      name: "addPet",
      displayName: "addPet",
      executor: (parameter: OperationTypes."/pets".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/pets".post.response200>, ResultFailure<HttpResponse<OperationTypes."/pets".post.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "POST",
            path: "/pets",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/pets".post.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/pets".post.responsedefault>)
          }
        }
    }
  },
  "/pets/offset": {
    get: {
      name: "findPetsOffset",
      displayName: "findPetsOffset",
      executor: (parameter: OperationTypes."/pets/offset".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/pets/offset".get.response200>, ResultFailure<HttpResponse<OperationTypes."/pets/offset".get.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "GET",
            path: "/pets/offset",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/pets/offset".get.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/pets/offset".get.responsedefault>)
          }
        }
    }
  },
  "/pets/page-number": {
    get: {
      name: "findPetsPageNumber",
      displayName: "findPetsPageNumber",
      executor: (parameter: OperationTypes."/pets/page-number".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/pets/page-number".get.response200>, ResultFailure<HttpResponse<OperationTypes."/pets/page-number".get.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "GET",
            path: "/pets/page-number",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/pets/page-number".get.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/pets/page-number".get.responsedefault>)
          }
        }
    }
  },
  "/pets/marker": {
    get: {
      name: "findPetsMarker",
      displayName: "findPetsMarker",
      executor: (parameter: OperationTypes."/pets/marker".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/pets/marker".get.response200>, ResultFailure<HttpResponse<OperationTypes."/pets/marker".get.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "GET",
            path: "/pets/marker",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/pets/marker".get.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/pets/marker".get.responsedefault>)
          }
        }
    }
  },
  "/pets/hypermedia": {
    get: {
      name: "findPetsHypermedia",
      displayName: "findPetsHypermedia",
      executor: (parameter: OperationTypes."/pets/hypermedia".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/pets/hypermedia".get.response200>, ResultFailure<HttpResponse<OperationTypes."/pets/hypermedia".get.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "GET",
            path: "/pets/hypermedia",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/pets/hypermedia".get.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/pets/hypermedia".get.responsedefault>)
          }
        }
    }
  },
  "/categories": {
      get: {
        name: "findCategories",
        displayName: "findCategories",
        executor: (parameter: OperationTypes."/categories".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/categories".get.response200>, ResultFailure<HttpResponse<OperationTypes."/categories".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/categories",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/categories".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/categories".get.responsedefault>)
            }
          }
      }
    },
  "/categories2": {
      get: {
        name: "findCategories2",
        displayName: "findCategories2",
        executor: (parameter: OperationTypes."/categories2".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/categories2".get.response200>, ResultFailure<HttpResponse<OperationTypes."/categories2".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/categories2",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/categories2".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/categories2".get.responsedefault>)
            }
          }
      }
    },
  "/categories3": {
      get: {
        name: "findCategories3",
        displayName: "findCategories3",
        executor: (parameter: OperationTypes."/categories3".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/categories3".get.response200>, ResultFailure<HttpResponse<OperationTypes."/categories3".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/categories3",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/categories3".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/categories3".get.responsedefault>)
            }
          }
      }
    },
  "/categories/{id}/tags": {
          get: {
            name: "findTags",
            displayName: "findTags",
            executor: (parameter: OperationTypes."/categories/{id}/tags".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/categories/{id}/tags".get.response200>, ResultFailure<HttpResponse<OperationTypes."/categories/{id}/tags".get.responsedefault>, Error>> -> do {
                var uri = parameter.uri
                var query = parameter.query default {}
                var headers = parameter.headers default {}
                var cookie = parameter.cookie default {}
                var response = connection({
                  method: "GET",
                  path: "/categories/$(uri.id)/tags",
                  queryParams: query,
                  headers: headers,
                  cookie: cookie,
                  (body: parameter.body) if (parameter.body?)
                })
                ---
                response.status match {
                  case 200 -> success(response as HttpResponse<OperationTypes."/categories/{id}/tags".get.response200>)
                  else -> failure(response as HttpResponse<OperationTypes."/categories/{id}/tags".get.responsedefault>)
                }
              }
          }
        },
  "/accounts" : {
      post: {
            name: "createAccount",
            displayName: "createAccount",
            executor: (parameter: OperationTypes."/account".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/account".post.response200>, ResultFailure<HttpResponse<OperationTypes."/account".post.responsedefault>, Error>> -> do {
                var uri = parameter.uri
                var query = parameter.query default {}
                var headers = parameter.headers default {}
                var cookie = parameter.cookie default {}
                var response = connection({
                  method: "POST",
                  path: "/account",
                  queryParams: query,
                  headers: headers,
                  cookie: cookie,
                  (body: parameter.body) if (parameter.body?)
                })
                ---
                response.status match {
                  case 200 -> success(response as HttpResponse<OperationTypes."/account".post.response200>)
                  else -> failure(response as HttpResponse<OperationTypes."/account".post.responsedefault>)
                }
              }
      }
  },
  "/record" : {
      post: {
            name: "createRecord",
            displayName: "createRecord",
            executor: (parameter: OperationTypes."/record".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/record".post.response200>, ResultFailure<HttpResponse<OperationTypes."/record".post.responsedefault>, Error>> -> do {
                var query = {}
                var headers = {}
                var cookie = {}
                var response = connection({
                  method: "POST",
                  path: "/$(parameter.recordType)",
                  queryParams: query,
                  headers: headers,
                  cookie: cookie,
                  (body: parameter.record) if (parameter.record?)
                })
                ---
                response.status match {
                  case 200 -> success(response as HttpResponse<OperationTypes."/record".post.response200>)
                  else -> failure(response as HttpResponse<OperationTypes."/record".post.responsedefault>)
                }
              }
      }
  },
  "/record-nested-medatadata-provider-param" : {
      post: {
            name: "createRecordMetadataFromNestedField",
            displayName: "createRecordMetadataFromNestedField",
            executor: (parameter: OperationTypes."/record-nested-medatadata-provider-param".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/record-nested-medatadata-provider-param".post.response200>, ResultFailure<HttpResponse<OperationTypes."/record-nested-medatadata-provider-param".post.responsedefault>, Error>> -> do {
                var query = {}
                var headers = {}
                var cookie = {}
                var response = connection({
                  method: "POST",
                  path: "/$(parameter.info.recordType)",
                  queryParams: query,
                  headers: headers,
                  cookie: cookie,
                  (body: parameter.record) if (parameter.record?)
                })
                ---
                response.status match {
                  case 200 -> success(response as HttpResponse<OperationTypes."/record-nested-medatadata-provider-param".post.response200>)
                  else -> failure(response as HttpResponse<OperationTypes."/record-nested-medatadata-provider-param".post.responsedefault>)
                }
              }
      }
  },
  "/order-custom-fields" : {
      post: {
            name: "createOrder",
            displayName: "createOrder",
            executor: (parameter: OperationTypes."/order-custom-fields".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/order-custom-fields".post.response200>, ResultFailure<HttpResponse<OperationTypes."/order-custom-fields".post.responsedefault>, Error>> -> do {
                var query = {}
                var headers = {}
                var cookie = {}
                var response = connection({
                  method: "POST",
                  path: "/order-custom-fields",
                  queryParams: query,
                  headers: headers,
                  cookie: cookie,
                  (body: parameter.body) if (parameter.body?)
                })
                ---
                response.status match {
                  case 200 -> success(response as HttpResponse<OperationTypes."/order-custom-fields".post.response200>)
                  else -> failure(response as HttpResponse<OperationTypes."/order-custom-fields".post.responsedefault>)
                }
              }
      }
  },
  "/account-metadata" : {
    get: {
        name: "accountMetadata",
        displayName: "accountMetadata",
        executor: (parameter: OperationTypes."/account-metadata".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/account-metadata".get.response200>, ResultFailure<HttpResponse<OperationTypes."/account-metadata".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/account-metadata",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/account-metadata".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/account-metadata".get.responsedefault>)
            }
          }
    }
  },
  "/tickets" : {
    get: {
          name: "getTickets",
          displayName: "getTickets",
          executor: (parameter: OperationTypes."/tickets".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/tickets".get.response200>, ResultFailure<HttpResponse<OperationTypes."/tickets".get.responsedefault>, Error>> -> do {
              var uri = parameter.uri
              var query = parameter.query default {}
              var headers = parameter.headers default {}
              var cookie = parameter.cookie default {}
              var response = connection({
                method: "GET",
                path: "/tickets",
                queryParams: query,
                headers: headers,
                cookie: cookie
              })
              ---
              response.status match {
                case 200 -> success(response as HttpResponse<OperationTypes."/tickets".get.response200>)
                else -> failure(response as HttpResponse<OperationTypes."/tickets".get.responsedefault>)
              }
            }
    }
    },
  "/ticket-metadata" : {
    get: {
        name: "ticketMetadata",
        displayName: "ticketMetadata",
        executor: (parameter: OperationTypes."/ticket-metadata".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/ticket-metadata".get.response200>, ResultFailure<HttpResponse<OperationTypes."/ticket-metadata".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/ticket-metadata",
              queryParams: query,
              headers: headers,
              cookie: cookie
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/ticket-metadata".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/ticket-metadata".get.responsedefault>)
            }
          }
    }
  },
  "/products/family/{id}/tags": {
      get: {
        name: "findProductTags",
        displayName: "findProductTags",
        executor: (parameter: OperationTypes."/products/family/{id}/tags".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/products/family/{id}/tags".get.response200>, ResultFailure<HttpResponse<OperationTypes."/products/family/{id}/tags".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/products/family/$(uri.id)/tags",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/products/family/{id}/tags".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/products/family/{id}/tags".get.responsedefault>)
            }
          }
      }
  },
  "/products/types/{id}/subTypes": {
      get: {
        name: "findProductSubTypes",
        displayName: "findProductSubTypes",
        executor: (parameter: OperationTypes."/products/types/{id}/subTypes".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/products/types/{id}/subTypes".get.response200>, ResultFailure<HttpResponse<OperationTypes."/products/types/{id}/subTypes".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/products/types/$(uri.id)/subTypes",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/products/types/{id}/subTypes".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/products/types/{id}/subTypes".get.responsedefault>)
            }
          }
      }
  },
  "/products/family/{id}/categories": {
      get: {
        name: "findProductCategories",
        displayName: "findProductCategories",
        executor: (parameter: OperationTypes."/products/family/{id}/categories".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/products/family/{id}/categories".get.response200>, ResultFailure<HttpResponse<OperationTypes."/products/family/{id}/categories".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/products/family/$(uri.id)/categories",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/products/family/{id}/categories".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/products/family/{id}/categories".get.responsedefault>)
            }
          }
      }
  },
  "/products": {
    post: {
        name: "addProduct",
        displayName: "addProduct",
        executor: (parameter: OperationTypes."/products".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/products".post.response200>, ResultFailure<HttpResponse<OperationTypes."/products".post.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "POST",
              path: "/products",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/products".post.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/products".post.responsedefault>)
            }
          }
      },
      get: {
        name: "findProducts",
        displayName: "findProducts",
        executor: (parameter: OperationTypes."/products".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/products".get.response200>, ResultFailure<HttpResponse<OperationTypes."/products".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/products",
              queryParams: query,
              headers: headers,
              cookie: cookie
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/products".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/products".get.responsedefault>)
            }
          }
      }
  },
  "/products/types": {
      get: {
        name: "findCategories",
        displayName: "findCategories",
        executor: (parameter: OperationTypes."/products/types".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/products/types".get.response200>, ResultFailure<HttpResponse<OperationTypes."/products/types".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/products/types",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/products/types".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/products/types".get.responsedefault>)
            }
          }
      }
    },
  "/clients" : {
        get: {
              name: "getClients",
              displayName: "getClients",
              executor: (parameter: OperationTypes."/clients".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/clients".get.response200>, ResultFailure<HttpResponse<OperationTypes."/clients".get.responsedefault>, Error>> -> do {
                  var uri = parameter.uri
                  var query = parameter.query default {}
                  var headers = parameter.headers default {}
                  var cookie = parameter.cookie default {}
                  var response = connection({
                    method: "GET",
                    path: "/clients",
                    queryParams: query,
                    headers: headers,
                    cookie: cookie,
                    (body: parameter.body) if (parameter.body?)
                  })
                  ---
                  response.status match {
                    case 200 -> success(response as HttpResponse<OperationTypes."/clients".get.response200>)
                    else -> failure(response as HttpResponse<OperationTypes."/clients".get.responsedefault>)
                  }
                }
        }
    },
  "/client-metadata" : {
    get: {
        name: "clientMetadata",
        displayName: "clientMetadata",
        executor: (parameter: OperationTypes."/client-metadata".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/client-metadata".get.response200>, ResultFailure<HttpResponse<OperationTypes."/client-metadata".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/client-metadata",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/client-metadata".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/client-metadata".get.responsedefault>)
            }
          }
    }
  },
  "/products/groups": {
      get: {
        name: "findGroups",
        displayName: "findGroups",
        executor: (parameter: OperationTypes."/products/groups".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/products/groups".get.response200>, ResultFailure<HttpResponse<OperationTypes."/products/groups".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/products/groups",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/products/groups".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/products/groups".get.responsedefault>)
            }
          }
      }
    },
  "/query": {
    get: {
      name: "queryBuilder",
      displayName: "queryBuilder",
      executor: (parameter: OperationTypes."/query".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/query".get.response200>, ResultFailure<HttpResponse<OperationTypes."/query".get.responsedefault>, Error>> -> do {
          var uri = parameter.uri
          var query = parameter.query default {}
          var headers = parameter.headers default {}
          var cookie = parameter.cookie default {}
          var response = connection({
            method: "GET",
            path: "/query",
            queryParams: query,
            headers: headers,
            cookie: cookie,
            (body: parameter.body) if (parameter.body?)
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/query".get.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/query".get.responsedefault>)
          }
        }
    }
  },
  "/queryBuilder-objects": {
      get: {
        name: "queryBuilderObjects",
        displayName: "queryBuilderObjects",
        executor: (parameter: OperationTypes."/queryBuilder-objects".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/queryBuilder-objects".get.response200>, ResultFailure<HttpResponse<OperationTypes."/queryBuilder-objects".get.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "GET",
              path: "/queryBuilder-objects",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/queryBuilder-objects".get.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/queryBuilder-objects".get.responsedefault>)
            }
          }
      }
    },
  "/queryBuilder-metadata" : {
      get: {
          name: "queryBuilderMetadata",
          displayName: "queryBuilderMetadata",
          executor: (parameter: OperationTypes."/queryBuilder-metadata".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/queryBuilder-metadata".get.response200>, ResultFailure<HttpResponse<OperationTypes."/queryBuilder-metadata".get.responsedefault>, Error>> -> do {
              var uri = parameter.uri
              var query = parameter.query default {}
              var headers = parameter.headers default {}
              var cookie = parameter.cookie default {}
              var response = connection({
                method: "GET",
                path: "/queryBuilder-metadata",
                queryParams: query,
                headers: headers,
                cookie: cookie
              })
              ---
              response.status match {
                case 200 -> success(response as HttpResponse<OperationTypes."/queryBuilder-metadata".get.response200>)
                else -> failure(response as HttpResponse<OperationTypes."/queryBuilder-metadata".get.responsedefault>)
              }
            }
      }
    },
"/all-types": {
    post: {
        name: "addAllTypes",
        displayName: "addAllTypes",
        executor: (parameter: OperationTypes."/all-types".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/all-types".post.response200>, ResultFailure<HttpResponse<OperationTypes."/all-types".post.responsedefault>, Error>> -> do {
            var uri = parameter.uri
            var query = parameter.query default {}
            var headers = parameter.headers default {}
            var cookie = parameter.cookie default {}
            var response = connection({
              method: "POST",
              path: "/all-types",
              queryParams: query,
              headers: headers,
              cookie: cookie,
              (body: parameter.body) if (parameter.body?)
            })
            ---
            response.status match {
              case 200 -> success(response as HttpResponse<OperationTypes."/all-types".post.response200>)
              else -> failure(response as HttpResponse<OperationTypes."/all-types".post.responsedefault>)
            }
          }
      }
  },
"/rest/api/3/attachment/content/{id}": {
  get: {
    name: "getAttachmentContent",
    displayName: "Get attachment content",
    executor: (parameter: OperationTypes."/rest/api/3/attachment/content/{id}".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/rest/api/3/attachment/content/{id}".get.response200>, ResultFailure<HttpResponse, Error>> -> do {
        var uri = parameter.uri
        var query = parameter.query default {}
        var headers = parameter.headers default {}
        var cookie = parameter.cookie default {}
        var response = connection({
          method: "GET",
          path: "/rest/api/3/attachment/content/$(uri.id)",
          queryParams: query,
          headers: headers,
          cookie: cookie,
          config: {
           followRedirects: true
          }
        })
        ---
        response.status match {
          case 200 -> success(response as HttpResponse<OperationTypes."/rest/api/3/attachment/content/{id}".get.response200>)
          else -> failure(response)
        }
      }
  }
},
"/rest/api/3/item/{resourceId}/attachments": {
  post: {
    name: "addAttachment",
    displayName: "Add attachment",
    executor: (parameter: OperationTypes."/rest/api/3/item/{resourceId}/attachments".post.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/rest/api/3/item/{resourceId}/attachments".post.response200>, ResultFailure<HttpResponse, Error>> -> do {
        var uri = parameter.uri
        var query = parameter.query default {}
        var headers = parameter.headers default {}
        var cookie = parameter.cookie default {}
        var response = connection({
          method: "POST",
          path: "/rest/api/3/item/TEST-123/attachments",
          queryParams: query,
          headers: headers,
          cookie: cookie,
          body: parameter.body
        })
        ---
        response.status match {
          case 200 -> success(response as HttpResponse<OperationTypes."/rest/api/3/item/{resourceId}/attachments".post.response200>)
          else -> failure(response)
        }
      }
    }
  },
  "/newstyle-errors": {
    get: {
      name: "Example operation (New errors!)",
      displayName: "Example operation (New errors!)",
      executor: (parameter: OperationTypes."/newstyle-errors".get.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/newstyle-errors".get.response200>, ResultFailure<Any, UnexpectedError> | ResultFailure<HttpResponse<OperationTypes."/newstyle-errors".get.responsedefault>, Default4XXError> | ResultFailure<HttpResponse<OperationTypes."/newstyle-errors".get.responsedefault>, Default5XXError>> -> do {
        var uri = parameter.uri
        var query = parameter.query default {}
        var headers = parameter.headers default {}
        var cookie = parameter.cookie default {}
        var response = connection({
          method: "GET",
          path: "/newstyle-errors",
          queryParams: query,
          headers: headers,
          cookie: cookie,
          body: parameter.body
        })
        var status = response.status as String
        ---
        response.status match {
          case 200 -> success(response as HttpResponse<OperationTypes."/newstyle-errors".get.response200>)
          case 404 -> default4XXFailure(response as HttpResponse<OperationTypes."/newstyle-errors".get.responsedefault>, {
            kind: "404",
            categories: ["client-error"]
          })
          case 502 -> default5XXFailure(response as HttpResponse<OperationTypes."/newstyle-errors".get.responsedefault>, {
            kind: "502",
            categories: ["server-error"]
          })
          else -> unexpectedFailure(response, {
            kind: status,
            categories: [status[0] match {
              case "4" -> "client-error"
              case "5" -> "server-error"
              else -> "unknown"
            }]
          })
        }
      }
    }
  }
}

// ----------------------------
// -- New Operation Versions --
// ----------------------------

type NewOperationTypes = {
  addPets: {
    request: {
      name: @Label(value = "Name of the Pet") String,
      category: @ValuesFrom(value = {name: "categoriesValueProvider"}) Number,
      tag: @ValuesFrom(value = { name: "tagsValueProvider", arguments: {id: "category" } }) String,
      age: Number
    }
  }
}

var newOperations = {
  addPets: {
      name: "addPet",
      displayName: "addPet",
      executor: (parameter: NewOperationTypes.addPets.request, connection: HttpConnection): Result<HttpResponse<OperationTypes."/pets".post.response200>, ResultFailure<HttpResponse<OperationTypes."/pets".post.responsedefault>, Error>> -> do {
          var response = connection({
            method: "POST",
            path: "/pets",
            queryParams: {},
            headers: {},
            cookie: {},
            body: parameter
          })
          ---
          response.status match {
            case 200 -> success(response as HttpResponse<OperationTypes."/pets".post.response200>)
            else -> failure(response as HttpResponse<OperationTypes."/pets".post.responsedefault>)
          }
      }
  }
}

// ---------------------
// -- Pagination --
// ---------------------

type PaginatedOperationTypes = {
  findPets: {
    request: {
        category?: @ValuesFrom(value = {name: "categoriesValueProvider"}) Number,
        limit?: Number
      }
  }
}

var paginatedOperations = {
  findPetsOffset: (offsetPaginated(
    operations."/pets/offset".get,
    (page) -> page.body default [],
    (param) -> param.query.from default 0,
    (param, offset) -> (param update { case from at .query.from! -> offset })
    ) mapInputPaginatedOperation (param: PaginatedOperationTypes.findPets.request) -> {query: {(category: param.category!) if (param.category?), (limit: param.limit!) if (param.limit?), (from: param.from as Number) if (param.from?)}, headers: {}, cookie: {}}),

  findPetsPageNumber: (pageNumberPaginated(
    operations."/pets/page-number".get,
    (page) -> page.body default [],
    (param) -> param.query.page default 0,
      (param, pageNumber) -> (param update { case page at .query.page! -> pageNumber })
    ) mapInputPaginatedOperation (param: PaginatedOperationTypes.findPets.request) -> {query: {(category: param.category!) if (param.category?), (limit: param.limit!) if (param.limit?), (page: param.page as Number) if (param.page?)}, headers: {}, cookie: {}}),

  findPetsMarker: (markerPaginated(
    operations."/pets/marker".get,
    (page) -> page.body.items default [],
    (page) -> page.body.nextToken default "",
      (param, marker) -> (param update { case markerToken at .query.markerToken! -> marker })
    ) mapInputPaginatedOperation (param: PaginatedOperationTypes.findPets.request) -> {query: {(category: param.category!) if (param.category?), (limit: param.limit!) if (param.limit?), (markerToken: param.markerToken as String) if (param.markerToken?)}, headers: {}, cookie: {}}),

  findPetsHypermedia: (hypermediaPaginated(
     operations."/pets/hypermedia".get,
     (page) -> page.body.items default [],
     (page) -> page.body.nextUrl default "",
     (param, nextUrl) -> param update { case query at .query -> parseQueryParameters(nextUrl default "")
                                         update {
                                           case limit at .limit! -> limit as Number default 0
                                           case from at .from! -> from as Number default 0
                                         }
                                 }) mapInputPaginatedOperation (param: PaginatedOperationTypes.findPets.request) -> {query: {(category: param.category!) if (param.category?), (limit: param.limit!) if (param.limit?), (from: param.from as Number) if (param.from?)}, headers: {}, cookie: {}}),

  findProductsGroups: (pageNumberPaginated(
      operations."/products/groups".get,
      (page) -> page.body default [],
      (param) -> param.query.page default 0,
        (param, pageNumber) -> (param update { case page at .query.page! -> pageNumber })
      ) mapInputPaginatedOperation (param: {}) -> {query: {(limit: param.query.limit as Number) if (param.query.limit?), (page: param.query.page as Number) if (param.query.page?)}, headers: {}, cookie: {}})
}

// ---------------------
// -- Value Providers --
// ---------------------

var categoriesValueProvider = defineValueProvider(
operations."/categories".get.executor mapInput (param: {}) -> {query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/categories".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

var categories2ValueProvider = defineValueProvider(
operations."/categories2".get.executor mapInput (param: {}) -> {query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/categories2".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

var categories3ValueProvider = defineValueProvider(
operations."/categories3".get.executor mapInput (param: {}) -> {query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/categories3".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

var tagsValueProvider = defineValueProvider(
operations."/categories/{id}/tags".get.executor mapInput (param: {id: Number}) -> {uri: {id: param.id}, query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/categories/{id}/tags".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

var productTagValueProvider = defineValueProvider(
operations."/products/family/{id}/tags".get.executor mapInput (param: {id: String}) -> {uri: {id: param.id}, query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/products/family/{id}/tags".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

var productSubTypeValueProvider = defineValueProvider(
operations."/products/types/{id}/subTypes".get.executor mapInput (param: {id: String}) -> {uri: {id: param.id}, query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/products/types/{id}/subTypes".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

var productCategoryValueProvider = defineValueProvider(
operations."/products/family/{id}/categories".get.executor mapInput (param: {id: String}) -> {uri: {id: param.id}, query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/products/family/{id}/categories".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])


var productTypeValueProvider = defineValueProvider(
operations."/products/types".get.executor mapInput (param: {}) -> {query: {}, headers: {}, cookie: {}},
	(response: HttpResponse<OperationTypes."/products/types".get.response200>) -> response.body map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

var productGroupValueProvider = definePaginatedValueProvider(
paginatedOperations.findProductsGroups.executor,
	(page: Page<Types.Tag, {}>) -> page.items map ((item, index) -> {value: item.id, displayValue: { label:item.label }})  default [])

// ---------------------
// -- Test Connection --
// ---------------------

@TestConnectionElement
var testConnection: HttpTestConnection = {
  validate: (connection) -> {
    isValid: connection({
      method: "GET",
      path: "/pets/offset",
      queryParams: {},
      headers: {},
      cookie: {}
    }).status == 200
  }
}

// ---------------------
// -- Metadata Providers --
// ---------------------
var accountDynamic : TypeValueDefinition =
  createObjectTypeValueDefinition("Account")

fun translateFields(fields : Array<Types.Field>) : Array<ObjectFieldValueDefinition> =
  do {
    fields map ((item, index) -> createField(item.name, translateField(item)))
  }

fun translateField(fieldDef: Types.Field) : TypeValueDefinition =
  do {
    if (fieldDef."type" == "Object")
      createObjectTypeValueDefinition(fieldDef.name) withFields translateFields(fieldDef.fields as Array<Types.Field> default [])
    else if (fieldDef."type" == "Number")
      createNumberTypeValueDefinition()
    else if (fieldDef."type" == "Boolean")
      createBooleanTypeValueDefinition()
    else if (fieldDef."type" == "Null")
      createNullTypeValueDefinition()
    else if (fieldDef."type" == "DateTime")
      createStringTypeValueDefinition() withFormat "date-time"
    else if (fieldDef."type" == "Date")
      createStringTypeValueDefinition() withFormat "date"
    else if (fieldDef."type" == "Array")
      //Not recursive, it's an example
      if (fieldDef.of? == "String")
        createArrayTypeValueDefinition(createStringTypeValueDefinition())
      else
        //Any other type is not supported yet
        createArrayTypeValueDefinition(createStringTypeValueDefinition())
    else if (fieldDef."type" == "Union")
      //Not recursive, it's an example
      createUnionTypeValueDefinition() of
        createStringTypeValueDefinition() of
        createNullTypeValueDefinition()
      else
        createStringTypeValueDefinition()
  }

var accountMetadataProvider: MetadataProviderExecutor<{}, Types.Error, HttpConnection> =
  (parameters, conn) -> do {
    var response =  operations."/account-metadata".get.executor({query: {}, headers: {}, cookie: {}}, conn)
    ---
    if (response.success and response.value.body != null)
      success((accountDynamic withFields translateFields(response.value.body as Array<Types.Field>)) as TypeValueDefinition)
    else
      failure(response.error.value.body as Types.Error)
  }

var recordMetadataProvider: MetadataProviderExecutor<{recordType: String}, Types.Error, HttpConnection> =
    (parameters, conn) -> do {
        if (parameters.recordType != "account")
            failure({
               code: 1,
               message: "error"
            })
        else
            accountMetadataProvider({}, conn)
    }

var ticketDynamic : TypeValueDefinition = createObjectTypeValueDefinition("Ticket")

var ticketMetadataProvider: MetadataProviderExecutor<{}, Types.Error, HttpConnection> =
  (parameters, conn) -> do {
    var response =  operations."/ticket-metadata".get.executor({query: {}, headers: {}, cookie: {}}, conn)
    ---
    if (response.success and response.value.body != null)
      success((ticketDynamic withFields translateFields(response.value.body as Array<Types.Field>)) as TypeValueDefinition)
    else
      failure(response.error.value.body as Types.Error)
  }

type Order = {
    id : Number,
    category : @ValuesFrom(value = {name: "categoriesValueProvider"}) String,
    tag?: @ValuesFrom(value = { name: "tagsValueProvider", arguments: {id: "category" } }) String,
    description: String,
    product?: Product
}

type Product = {
    id : Number,
    name : String,
    description: String
}

var orderMetadataProvider: MetadataProviderExecutor<{}, Types.Error, HttpConnection> =
  (parameters, conn) -> success(
    createExtendedObjectTypeValueDefinition("Order", Order)
        withField createField("customField", createNumberTypeValueDefinition())
    )

var clientDynamic : TypeValueDefinition = createObjectTypeValueDefinition("Client")

var clientMetadataProvider: MetadataProviderExecutor<{}, Types.Error, HttpConnection> =
  (parameters, conn) -> do {
    var response =  operations."/client-metadata".get.executor({query: {}, headers: {}, cookie: {}}, conn)
    ---
    if (response.success and response.value.body != null)
      success((clientDynamic withFields translateFields(response.value.body as Array<Types.Field>)) as TypeValueDefinition)
    else
      failure(response.error.value.body as Types.Error)
  }

// ---------------------
// -- Triggers --
// ---------------------

type OnNewPetTriggerUserInitWatermarkInput = {
                                                category?: @ValuesFrom(value = {name: "categoriesValueProvider"}) Number,
                                                limit?: Number,
                                                startDate?: LWDate
                                            }

var onNewCategoryTriggerStrategy: TriggerStrategy<HttpResponse<OperationTypes."/categories".get.response200>, Types.Category, Types.Category, Number> = {
    items: (result) -> result.body!,
    item: (item) -> item,
    watermark: (result, item) -> item.id,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

var onNewPetTriggerStrategy: TriggerStrategy<Page<Types.Pet, OperationTypes."/pets/offset".get.request>,
 Types.Pet, Types.Pet, DateTime> = {
    items: (result) -> result.items!,
    item: (item) -> item,
    watermark: (result, item) -> (item.createdAt! ++ "T10:00:00.100-00:00") as DateTime ,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

var onNewTicketTriggerStrategy: TriggerStrategy<HttpResponse<OperationTypes."/tickets".get.response200>, Types.Ticket, Types.Ticket, Number> = {
    items: (result) -> result.body!,
    item: (item) -> item,
    watermark: (result, item) -> item.id,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

var onNewClientTriggerStrategy: TriggerStrategy<HttpResponse<OperationTypes."/clients".get.response200>, Types.Client, Types.Client, Number> = {
    items: (result) -> result.body!,
    item: (item) -> item,
    watermark: (result, item) -> item.id,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

var onNewProductTriggerStrategy: TriggerStrategy<HttpResponse<OperationTypes."/products".get.response200>, Types.Product, Types.Product, Number> = {
    items: (result) -> result.body!,
    item: (item) -> item,
    watermark: (result, item) -> item.id,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}


var triggers = {
  onNewCategoryTrigger: {
      name: "onNewCategoryTrigger",
      displayName: "On new category Trigger",
      metadata: {
          order: "ASC",
          paginated: false
      },
      strategy: onNewCategoryTriggerStrategy,
      operation: operations."/categories".get,
      inputMapper: (param: {}, watermark: Number) -> {query: {}, headers: {}, cookie: {}},
      initialWatermark: (triggerInput, connection) -> 1
  },
  onNewPetTrigger: {
      name: "onNewPetTrigger",
      displayName: "On new pet Trigger",
      metadata: {
          order: "ASC",
          paginated: true
      },
      strategy: onNewPetTriggerStrategy,
      operation: paginatedOperations.findPetsOffset,
      inputMapper: (ti: PaginatedOperationTypes.findPets.request, w) -> ti,
      initialWatermark: (triggerInput, connection) -> "2024-01-01T10:00:00.100-00:00" as DateTime
  },
  onNewTicketTrigger: {
      name: "onNewTicketTrigger",
      displayName: "On new ticket Trigger",
      metadata: {
          order: "ASC",
          paginated: false
      },
      strategy: onNewTicketTriggerStrategy,
      operation: operations."/tickets".get,
      initialWatermark: (triggerInput, connection) -> 1,
      inputMapper: (param:  { //Input just for vp test, it make no sense in this trigger operation
                                category?: @ValuesFrom(value = {name: "categoriesValueProvider"}) @Label(value = "Category ID") @Description(value = "The category of the trigger") Number,
                                tag?: @ValuesFrom(value = { name: "tagsValueProvider", arguments: {id: "category" } }) String
                            }, watermark: Number) -> {query: {}, headers: {}, cookie: {}}
  },
  onNewProductTrigger: {
        name: "onNewProductTrigger",
        displayName: "On new product Trigger",
        metadata: {
            order: "ASC",
            paginated: false
        },
        strategy: onNewProductTriggerStrategy,
        operation: operations."/products".get,
        initialWatermark: (triggerInput, connection) -> 1,
        inputMapper: (param: {
            productType?: @ValuesFrom(value = {name: "productTypeValueProvider"}) @Label(value = "Product Type") @Description(value = "The product type of the trigger") String
          }, watermark: Number) -> {query: {}, headers: {}, cookie: {}}
    },
  onNewClientTrigger: {
        name: "onNewClientTrigger",
        displayName: "On new client Trigger",
        metadata: {
            order: "ASC",
            paginated: false
        },
        strategy: onNewClientTriggerStrategy,
        operation: operations."/clients".get,
        inputMapper: (param: {}, watermark: Number) -> {query: {}, headers: {}, cookie: {}},
        initialWatermark: (triggerInput, connection) -> 1
    },
  onNewPetTriggerUserInitWatermark: {
      name: "onNewPetTriggerUserInitWatermark",
      displayName: "On new pet Trigger user init watermark",
      metadata: {
          order: "ASC",
          paginated: true
      },
      strategy: onNewPetTriggerStrategy,
      operation: paginatedOperations.findPetsOffset,
      inputMapper: (ti: OnNewPetTriggerUserInitWatermarkInput, w) -> {
                            (category: ti.category) if ti.category?,
                            (limit: ti.limit) if ti.limit?
                          } as PaginatedOperationTypes.findPets.request,
      initialWatermark: (triggerInput, connection) -> triggerInput.startDate as DateTime default |2024-01-01T10:00:00.1Z|
  }
}

// --------------------------
// -- New Trigger Versions --
// --------------------------

var newTriggers = {
  onNewPetTrigger: {
      name: "onNewPetTrigger",
      displayName: "On new pet Trigger",
      metadata: {
          order: "ASC",
          paginated: true
      },
      strategy: onNewPetTriggerStrategy,
      operation: paginatedOperations.findPetsOffset mapInputPaginatedOperation (param: PaginatedOperationTypes.findPets.request & {newDummyParam?: String}) -> param,
      inputMapper: (ti: PaginatedOperationTypes.findPets.request & {newDummyParam?: String}, w) -> ti,
      initialWatermark: (triggerInput, connection) -> "2024-01-01T10:00:00.100-00:00" as DateTime
  }
}

