/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.sdk.api.annotation.param.RefName;

import javax.inject.Inject;

import static java.lang.String.format;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class RestConnectionProvider<C extends RestConnection> implements CachedConnectionProvider<C>,
    Startable,
    Stoppable {

  /**
   * A mask pattern for building the name of the {@link HttpClient} that will be created to serve this provider.
   */
  static final String CLIENT_NAME_PATTERN = "rest.connector.%s";

  /**
   * The name of the config in which this connection is defined
   */
  @RefName
  private String configName;

  @Inject
  private HttpService httpService;

  private HttpClient httpClient;

  /**
   * @return The base uri that will be used for all HTTP requests.
   */
  public abstract String getBaseUri();

  @Override
  public final C connect() throws ConnectionException {
    try {
      return createConnection(httpClient);
    } catch (Exception e) {
      throw new ConnectionException("Could not create connection", e);
    }
  }

  /**
   * Creates a new connection {@link C}
   *
   * @param httpClient the client to perform the requests
   * @return a new {@link C}
   */
  // TODO why httpClient is a parameter when the baseUri is a getter?. Review.
  protected abstract C createConnection(HttpClient httpClient);

  @Override
  public final void disconnect(C connection) {}

  @Override
  public ConnectionValidationResult validate(C connection) {
    // TODO I'd vote to make this method abstract and force an implementation.
    return ConnectionValidationResult.success();
  }

  /**
   * Starts the resources related to this connection provider.
   */
  @Override
  public void start() {
    startHttpClient();
  }

  /**
   * Stops the started resources related to this connection provider.
   */
  @Override
  public void stop() {
    if (httpClient != null) {
      httpClient.stop();
    }
  }

  /**
   * Initialises the http client that will be used by this connection provider to create new connections. Besides all the standard
   * configuration that is initialized here, the {@link #configureClient(HttpClientConfiguration.Builder)} method is provided in
   * order for extending classes to set its custom configurations.
   */
  private void startHttpClient() {
    HttpClientConfiguration.Builder configuration = new HttpClientConfiguration.Builder()
        .setName(format(CLIENT_NAME_PATTERN, configName));

    configureClient(configuration);

    httpClient = httpService.getClientFactory().create(configuration.build());
    httpClient.start();
  }

  /**
   * Template method that will be invoked just before the {@code httpConfiguration} is turned into an actual {@link HttpClient}.
   * It gives implementations a chance to do extra configurations.
   *
   * @param httpConfiguration the configuration builder for the {@link HttpClient} about to be created.
   */
  protected void configureClient(HttpClientConfiguration.Builder httpConfiguration) {
    // no-op by default
  }
}
