/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.metadatamodel.handler;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.IntersectionTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;

import java.util.ArrayList;
import java.util.Collection;

import org.everit.json.schema.CombinedSchema;
import org.everit.json.schema.EnumSchema;
import org.everit.json.schema.NullSchema;
import org.everit.json.schema.Schema;
import org.everit.json.schema.StringSchema;

public class CombinedHandler implements SchemaHandler {

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof CombinedSchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, RestHandlerManager handlerManager,
                               ParsingContext parsingContext) {
    final CombinedSchema combinedSchema = (CombinedSchema) schema;
    final Collection<Schema> subSchemas = combinedSchema.getSubschemas();
    if (combinedSchema.getCriterion() == CombinedSchema.ANY_CRITERION
        || combinedSchema.getCriterion() == CombinedSchema.ONE_CRITERION) {

      ArrayList<Schema> uniqueSubSchemas = new ArrayList<>();
      if (subSchemas.size() == 1) {
        return handlerManager.handle(subSchemas.stream().findFirst().get(), parsingContext);
      } else {
        for (Schema subSchema : subSchemas) {
          if (!(subSchema instanceof NullSchema)) {
            uniqueSubSchemas.add(subSchema);
          }
        }
        if (uniqueSubSchemas.size() == 1) {
          return handlerManager.handle(uniqueSubSchemas.get(0), parsingContext);
        }
        final UnionTypeBuilder unionType = root.unionType();
        for (Schema subSchema : uniqueSubSchemas) {
          unionType.of(handlerManager.handle(subSchema, parsingContext));
        }
        return unionType;
      }
    } else if (combinedSchema.getCriterion() == CombinedSchema.ALL_CRITERION) {
      final TypeBuilder typeBuilder;
      ArrayList<Schema> uniqueSubSchemas = new ArrayList<>();

      if (subSchemas.size() == 2) {
        final boolean enumSchemaPresent =
            subSchemas.stream().anyMatch(currentSubSchema -> currentSubSchema instanceof EnumSchema);
        final boolean stringSchemaPresent =
            subSchemas.stream().anyMatch(currentSubSchema -> currentSubSchema instanceof StringSchema);
        // Only if "type": "string" and "enum" is defined we process enum schema, for the other
        // scenarios number/boolean/integer with enum we discard the enum schema and use the base type.
        for (Schema subSchema : subSchemas) {
          if ((enumSchemaPresent && stringSchemaPresent && subSchema instanceof EnumSchema) ||
              (enumSchemaPresent && !stringSchemaPresent && !(subSchema instanceof EnumSchema))) {
            uniqueSubSchemas.add(subSchema);
          }
        }
        // If we don't have the expected combination of types with enum just add every subSchema to build an intersection
        if (uniqueSubSchemas.isEmpty()) {
          uniqueSubSchemas.addAll(subSchemas);
        }
      } else {
        uniqueSubSchemas.addAll(subSchemas);
      }

      if (uniqueSubSchemas.size() == 1) {
        typeBuilder = handlerManager.handle(uniqueSubSchemas.iterator().next(), parsingContext);
      } else if (uniqueSubSchemas.size() == 0) {
        typeBuilder = root.anyType();
      } else {
        final IntersectionTypeBuilder intersectionType = root.intersectionType();
        for (Schema subSchema : uniqueSubSchemas) {
          intersectionType.of(handlerManager.handle(subSchema, parsingContext));
        }
        typeBuilder = intersectionType;
      }
      return typeBuilder;
    }
    return root.anyType();
  }
}
