/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

public class DefaultRestConnection implements RestConnection {

  private final HttpClient httpClient;
  private final HttpRequestOptions httpRequestOptions;
  private final String baseUri;

  public DefaultRestConnection(HttpClient httpClient, HttpRequestOptions httpRequestOptions, String baseUri) {
    this.httpClient = httpClient;
    this.httpRequestOptions = httpRequestOptions;
    this.baseUri = baseUri;
  }

  private HttpClient getHttpClient() {
    return httpClient;
  }

  private HttpRequestOptions getHttpRequestOptions() {
    return httpRequestOptions;
  }

  public String getBaseUri() {
    return baseUri;
  }

  @Override
  public CompletableFuture<HttpResponse> sendAsync(HttpRequest request) {
    return getHttpClient().sendAsync(beforeRequest(request), getHttpRequestOptions());
  }

  @Override
  public HttpResponse send(HttpRequest request) throws IOException, TimeoutException {
    return getHttpClient().send(beforeRequest(request), getHttpRequestOptions());
  }

  @Override
  public ConnectionValidationResult validate(HttpRequest request,
                                             Function<HttpResponse, ConnectionValidationResult> whenComplete,
                                             Function<Exception, ConnectionValidationResult> onError) {
    try {
      return whenComplete.apply(getHttpClient().send(beforeRequest(request), getHttpRequestOptions()));
    } catch (Exception e) {
      return onError.apply(e);
    }
  }

  private HttpRequest beforeRequest(HttpRequest request) {
    HttpRequestBuilder httpRequestBuilder = HttpRequest.builder()
        .method(request.getMethod())
        .protocol(request.getProtocol())
        .uri(request.getUri())
        .queryParams(request.getQueryParams())
        .headers(request.getHeaders())
        .entity(request.getEntity());

    authenticate(httpRequestBuilder);

    return httpRequestBuilder.build();
  }

  /**
   * Allows to different type of connections to intercept the request before sending the HTTP call. This would be useful for
   * Bearer, API Key or OAuth connections which need to include headers for authentication.
   *
   * @param httpRequestBuilder {@link HttpRequestBuilder} populated with the values from the original {@link HttpRequest}.
   * @return either the same request or one modified.
   */
  protected void authenticate(HttpRequestBuilder httpRequestBuilder) {
    // Nothing to do by default
  }

}
