/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants.PAYLOAD;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestUtils.toList;
import static com.mulesoft.connectivity.rest.commons.internal.util.SplitPayloadUtils.split;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy.PagingStrategy;

import java.util.Iterator;
import java.util.List;
import java.util.function.Function;

/**
 * An implementation for {@link RestPagingProvider<C>} which uses DataWeave to extract the payload and create the list of pages.
 * <p>
 * Type parameters: <C> – subclass of {@link RestConnection}
 * 
 * @since 1.0
 */
public class RestPagingDataWeaveProvider<C extends RestConnection> extends RestPagingProvider<C> {

  private final ExpressionLanguage expressionLanguage;

  private final String pageItemsExtractorExpression;

  /**
   * Creates a new instance
   *
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance.
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one.
   * @param pagingStrategy the pagination strategy.
   * @param expressionLanguage {@link ExpressionLanguage} to evaluate the {@link #pageItemsExtractorExpression} to extract the
   *        pages content from the server's response.
   * @param pageItemsExtractorExpression a DataWeave expression to extract the response with the page data.
   */
  public RestPagingDataWeaveProvider(Function<C, HttpRequestBuilder> requestFactory,
                                     MediaType defaultMediaType,
                                     PagingStrategy pagingStrategy,
                                     ExpressionLanguage expressionLanguage,
                                     String pageItemsExtractorExpression) {
    super(requestFactory, defaultMediaType, pagingStrategy);
    checkArgument(expressionLanguage != null, "ExpressionLanguage cannot be null");
    checkArgument(isNotBlank(pageItemsExtractorExpression), "pageItemsExtractorExpression cannot be blank");

    this.expressionLanguage = expressionLanguage;
    this.pageItemsExtractorExpression = pageItemsExtractorExpression;
  }

  @Override
  protected List<TypedValue<String>> extractPageItems(TypedValue<String> content, MediaType defaultMediaType) {
    TypedValue<?> payload;
    try {
      payload = expressionLanguage.evaluate(pageItemsExtractorExpression, content.getDataType(),
                                            BindingContext.builder()
                                                .addBinding(PAYLOAD, content)
                                                .build());
    } catch (ExpressionRuntimeException e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to extract payload from expression: %s",
                                                                pageItemsExtractorExpression)),
                                     e);
    }
    Iterator<TypedValue<?>> iterator = split(expressionLanguage, payload);
    return toList(iterator, content.getDataType().getMediaType(), defaultMediaType);
  }

}
