/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy;

import static com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants.HEADERS;
import static com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants.LINK;
import static com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants.PAYLOAD;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;

/**
 * An implementation for {@link HypermediaPagingStrategy} which uses DataWeave to extract the next URL.
 *
 * @since 1.0
 */
public class DataweaveHypermediaPagingStrategy extends HypermediaPagingStrategy {

  private final String nextUrlExpression;
  private final ExpressionLanguage expressionLanguage;

  public DataweaveHypermediaPagingStrategy(String nextUrlExpression, ExpressionLanguage expressionLanguage) {
    checkArgument(expressionLanguage != null, "ExpressionLanguage cannot be null");
    checkArgument(isNotBlank(nextUrlExpression), "nextUrlExpression cannot be blank");

    this.nextUrlExpression = nextUrlExpression;
    this.expressionLanguage = expressionLanguage;
  }

  @Override
  protected String extractNextUrl(TypedValue<String> rawPage, MultiMap<String, String> headers) {
    try {
      return (String) expressionLanguage.evaluate(nextUrlExpression, DataType.STRING,
                                                  BindingContext.builder()
                                                      .addBinding(PAYLOAD, rawPage)
                                                      .addBinding(HEADERS, TypedValue.of(headers))
                                                      .addBinding(LINK, TypedValue.of(buildLinkHeaderMap(headers.get(LINK))))
                                                      .build())
          .getValue();
    } catch (ExpressionRuntimeException e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to extract next url from expression: %s",
                                                                nextUrlExpression)),
                                     e);
    }
  }
}
