/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestUtils.stringValue;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import com.mulesoft.connectivity.rest.commons.api.operation.paging.RestPagingProvider;
import com.mulesoft.connectivity.rest.commons.api.operation.paging.RestPagingProvider.PagingContext;

import java.util.List;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on an offset query param.
 *
 * @since 1.0
 */
public class OffsetPagingStrategy implements PagingStrategy {

  private final String offsetParamName;
  private int offset;

  /**
   * Creates a new instance
   *
   * @param offsetParamName the name of the query param that carries the offset value
   * @param initialOffset the initial offset to apply on the first request
   */
  public OffsetPagingStrategy(String offsetParamName, int initialOffset) {
    checkArgument(isNotBlank(offsetParamName), "offsetParamName cannot be blank");

    this.offsetParamName = offsetParamName;
    offset = initialOffset;
  }

  @Override
  public final HttpRequestBuilder configureRequest(String baseUri, HttpRequestBuilder requestBuilder, PagingContext context) {
    requestBuilder.addQueryParam(offsetParamName, stringValue(offset));
    return requestBuilder;
  }

  @Override
  public final void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage, MultiMap<String, String> headers,
                           PagingContext context) {
    offset += page.size();
  }

}
