/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestUtils.stringValue;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import com.mulesoft.connectivity.rest.commons.api.operation.paging.RestPagingProvider.PagingContext;

import java.util.List;

/**
 * Base class of {@link PagingStrategy} for APIs which do paging based on requesting individual pages by number
 *
 * @since 1.0
 */
public class PageNumberPagingStrategy implements PagingStrategy {

  private final String pageNumberParamName;
  private int pageNumber;

  /**
   * Creates a new instance
   *
   * @param pageNumberParamName the name of the query param that carries the page number value
   * @param initialPageNumber the initial page to get in the first request
   */
  public PageNumberPagingStrategy(String pageNumberParamName, int initialPageNumber) {
    checkArgument(isNotBlank(pageNumberParamName), "pageNumberParamName cannot be blank");

    this.pageNumberParamName = pageNumberParamName;
    this.pageNumber = initialPageNumber;
  }

  @Override
  public HttpRequestBuilder configureRequest(String baseUri, HttpRequestBuilder requestBuilder, PagingContext context) {
    requestBuilder.addQueryParam(pageNumberParamName, stringValue(pageNumber));
    return requestBuilder;
  }

  @Override
  public void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage,
                     MultiMap<String, String> headers, PagingContext context) {
    Integer totalPages = extractPageCount(rawPage, headers);
    if (totalPages != null && pageNumber == totalPages) {
      context.stopPaging();
    } else {
      pageNumber++;
    }
  }

  public Integer extractPageCount(TypedValue<String> rawPage, MultiMap<String, String> headers) {
    return null;
  }

}
