/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static java.lang.Integer.MAX_VALUE;
import static java.lang.String.valueOf;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Optional.of;
import static org.mule.runtime.core.api.functional.Either.left;
import static org.mule.runtime.core.api.functional.Either.right;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;

import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.CursorProvider;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.api.functional.Either;
import org.mule.runtime.core.api.util.IOUtils;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * Utility methods
 *
 * @since 1.0
 */
public final class RestUtils {

  private RestUtils() {}

  /***
   * Closes the given {@code stream} which can be either a {@link CursorProvider} or an {@link InputStream}.
   *
   * Null values or instances of other classes will be ignored.
   *
   * @param stream a stream
   */
  public static void closeStream(Object stream) {
    if (stream instanceof CursorProvider) {
      CursorProvider cursorProvider = ((CursorProvider) stream);
      cursorProvider.close();
      cursorProvider.releaseResources();
    } else if (stream instanceof InputStream) {
      closeQuietly((InputStream) stream);
    }
  }

  /**
   * Consumes the contents of the given {@code stream} as a String and closes it.
   * <p>
   * The {@code stream} can be either a {@link CursorStreamProvider} or an {@link InputStream}. {@link IllegalArgumentException}
   * will be thrown if the {@code stream} is an instance of any other class.
   * <p>
   * Notice that in the case of {@link CursorStreamProvider}, this method will close the provider meaning that it will no longer
   * be able to yield new cursors.
   *
   * @param stream a {@link CursorStreamProvider} or an {@link InputStream}
   * @param targetCharset the encoding to use when reading the String and that will be set to the TypedValue
   * @param targetMediaType the media type that will be set to the TypedValue
   * @return the contents as a String
   * @throws IllegalArgumentException if {@code stream} is not of the expected types.
   */
  public static TypedValue<String> consumeStringAndClose(Object stream, MediaType targetMediaType, Charset targetCharset) {
    return consumeStringTransformAndClose(stream, targetCharset, targetMediaType, targetCharset);
  }

  /**
   * Transform an iterator of TypedValue<?> into a list of TypedValue<String> consuming the iterator typed values using the
   * provided media type and charset.
   *
   * @param iterator The iterator to consume
   * @param mediaType The primary source of the media type and charset. This is the media type for this specific iterator.
   * @param defaultMediaType Tee default source for the charset. The charset of this media type will be used if the primary one
   *        does not contain one.
   */
  public static List<TypedValue<String>> toList(Iterator<TypedValue<?>> iterator, MediaType mediaType,
                                                MediaType defaultMediaType) {
    List<TypedValue<String>> list = new ArrayList<>();

    Charset encoding = resolveCharset(of(mediaType), defaultMediaType);

    iterator.forEachRemaining(v -> {
      TypedValue<String> stringTypedValue;

      if (v.getDataType().getMediaType().getCharset().isPresent()
          && v.getDataType().getMediaType().getCharset().get().equals(encoding)) {
        stringTypedValue = consumeStringAndClose(v.getValue(), mediaType, encoding);
      } else {
        stringTypedValue =
            consumeStringTransformAndClose(v.getValue(), v.getDataType().getMediaType().getCharset().get(), mediaType, encoding);
      }

      list.add(stringTypedValue);
    });

    return list;
  }

  /***
   * Transforms a value into a tailored String representation for HTTP requests.
   *
   * @return String representation of the parameter
   */
  public static String stringValue(Object o) {
    if (o == null) {
      return null;
    }

    if (o instanceof String) {
      return (String) o;
    }

    if (o instanceof Integer | o instanceof Long | o instanceof Double) {
      NumberFormat numberFormat = NumberFormat.getInstance();
      numberFormat.setMaximumFractionDigits(MAX_VALUE);
      numberFormat.setGroupingUsed(false);
      return numberFormat.format(o);
    }

    return valueOf(o);
  }

  private static Charset resolveCharset(Optional<MediaType> mediaType, MediaType defaultMediaType) {
    return mediaType
        .flatMap(MediaType::getCharset)
        .orElseGet(() -> defaultMediaType.getCharset().orElse(defaultCharset()));
  }

  /**
   * Consumes the contents of the given {@code stream} as a String and closes it.
   * <p>
   * The {@code stream} can be either a {@link CursorStreamProvider} or an {@link InputStream}. {@link IllegalArgumentException}
   * will be thrown if the {@code stream} is an instance of any other class.
   * <p>
   * Notice that in the case of {@link CursorStreamProvider}, this method will close the provider meaning that it will no longer
   * be able to yield new cursors.
   *
   * @param stream a {@link CursorStreamProvider} or an {@link InputStream}
   * @param sourceCharset the encoding to use when reading the String
   * @param targetCharset the encoding that will be set to the TypedValue
   * @param targetMediaType the media type that will be set to the TypedValue
   * @return the contents as a String
   * @throws IllegalArgumentException if {@code stream} is not of the expected types.
   */
  private static TypedValue<String> consumeStringTransformAndClose(Object stream, Charset sourceCharset,
                                                                   MediaType targetMediaType,
                                                                   Charset targetCharset) {
    if (stream == null) {
      return toTypedValue("", targetMediaType, targetCharset);
    }

    if (stream instanceof String) {
      return toTypedValue((String) stream, targetMediaType, targetCharset);
    }

    Either<CursorStreamProvider, InputStream> content;

    if (stream instanceof CursorStreamProvider) {
      content = left((CursorStreamProvider) stream);
    } else if (stream instanceof InputStream) {
      content = right((InputStream) stream);
    } else {
      throw new IllegalArgumentException("Cannot consume stream of unsupported type: " + stream.getClass().getName());
    }

    return content.reduce(provider -> {
      try {
        return doConsumeAndClose(provider.openCursor(), sourceCharset, targetMediaType, targetCharset);
      } finally {
        closeStream(provider);
      }
    }, in -> doConsumeAndClose(in, sourceCharset, targetMediaType, targetCharset));
  }

  private static TypedValue<String> doConsumeAndClose(InputStream stream, Charset sourceCharset, MediaType targetMediaType,
                                                      Charset targetCharset) {
    try {
      return toTypedValue(IOUtils.toString(stream, sourceCharset), targetMediaType, targetCharset);
    } finally {
      closeQuietly(stream);
    }
  }

  public static TypedValue<String> toTypedValue(String value, MediaType mediaType, Charset encoding) {
    return new TypedValue<>(value, DataType.builder().mediaType(mediaType).charset(encoding).build());
  }

}
