/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants.PAYLOAD;
import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.BOOLEAN;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_XML;
import static org.mule.runtime.api.metadata.MediaType.XML;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.TypedValue;

import java.util.Iterator;

/**
 * Utils class for splitting payloads.
 *
 * @since 1.0
 */
public class SplitPayloadUtils {

  /**
   * Splits a payload handling its media type and the needed expressions automatically.
   * 
   * @param expressionLanguage The instance of the expression language to use.
   * @param payload The payload to split.
   * @return An iterator containing an element for each item in the split payload.
   */
  public static Iterator<TypedValue<?>> split(ExpressionLanguage expressionLanguage, TypedValue<?> payload) {
    Iterator<TypedValue<?>> splitResult;
    if (isXml(payload)) {
      validatePayload(expressionLanguage, payload, "payload is Object");
      splitResult = expressionLanguage.split("payload[0] default []", buildContext(payload));
    } else {
      validatePayload(expressionLanguage, payload, "payload is Array");
      splitResult = expressionLanguage.split("payload", buildContext(payload));
    }
    return splitResult;
  }

  private static boolean isXml(TypedValue<?> payload) {
    return payload.getDataType().getMediaType().matches(APPLICATION_XML) || payload.getDataType().getMediaType().matches(XML);
  }

  private static void validatePayload(ExpressionLanguage expressionLanguage,
                                      TypedValue<?> payload,
                                      String expression) {
    TypedValue<?> result = expressionLanguage.evaluate(expression, BOOLEAN, buildContext(payload));
    if (result.getValue().equals(false)) {
      throw new MuleRuntimeException(createStaticMessage(format("Failed to split payload that is not an array. Result Payload: '%s'.",
                                                                payload.getValue())));
    }
  }


  private static BindingContext buildContext(TypedValue<?> payload) {
    return BindingContext.builder()
        .addBinding(PAYLOAD, payload).build();
  }

}
