/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;

import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.client.HttpRequestOptionsBuilder;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.sdk.api.annotation.param.RefName;

import java.util.Optional;

import javax.inject.Inject;

public abstract class RestConnectionProvider<C extends RestConnection> implements CachedConnectionProvider<C>, Initialisable,
    Startable, Stoppable {

  /**
   * A mask pattern for building the name of the {@link HttpClient} that will be created to serve this provider.
   */
  static final String CLIENT_NAME_PATTERN = "rest.connector.%s";

  /**
   * The name of the config in which this connection is defined
   */
  @RefName
  private String configName;

  @Inject
  protected HttpService httpService;

  private HttpClient httpClient;

  private HttpAuthentication authentication;

  /**
   * @return The base uri that will be used for all HTTP requests.
   */
  public abstract String getBaseUri();

  @Override
  public final C connect() throws ConnectionException {
    try {
      return createConnection(httpClient, getHttpRequestOptions());
    } catch (ConnectionException e) {
      throw e;
    } catch (Exception e) {
      throw new ConnectionException("Could not create connection", e);
    }
  }

  /**
   * Creates a new connection {@link C}
   *
   * @param httpClient the client to perform the requests.
   * @param httpRequestOptions the options for the http request client to perfom the requests.
   * @return a new {@link C}
   * @throws {@link ConnectionException} if there was an error while creating the connection.
   */
  protected abstract C createConnection(HttpClient httpClient, HttpRequestOptions httpRequestOptions) throws ConnectionException;

  @Override
  public final void disconnect(C connection) {}

  @Override
  public void initialise() throws InitialisationException {
    authentication = buildAuthentication();
    if (authentication != null) {
      initialiseIfNeeded(authentication);
    }
  }

  /**
   * Starts the resources related to this connection provider.
   */
  @Override
  public void start() {
    startHttpClient();
  }

  /**
   * Stops the started resources related to this connection provider.
   */
  @Override
  public void stop() {
    if (httpClient != null) {
      httpClient.stop();
    }
  }

  /**
   * Initialises the http client that will be used by this connection provider to create new connections. Besides all the standard
   * configuration that is initialized here, the {@link #configureClient(HttpClientConfiguration.Builder)} method is provided in
   * order for extending classes to set its custom configurations.
   */
  private void startHttpClient() {
    HttpClientConfiguration.Builder configuration = new HttpClientConfiguration.Builder()
        .setName(format(CLIENT_NAME_PATTERN, configName));

    configureClient(configuration);

    httpClient = httpService.getClientFactory().create(configuration.build());
    httpClient.start();
  }

  /**
   * Template method that will be invoked just before the {@code httpConfiguration} is turned into an actual {@link HttpClient}.
   * It gives implementations a chance to do extra configurations.
   *
   * @param httpConfiguration the configuration builder for the {@link HttpClient} about to be created.
   */
  protected void configureClient(HttpClientConfiguration.Builder httpConfiguration) {
    // no-op by default
  }

  /**
   * @return the {@link HttpAuthentication} defined by the connection provider or {@code null}. This allows implementations to
   *         support custom authentications by defining at the implementation class the parameters for filling the authentication.
   */
  protected HttpAuthentication buildAuthentication() {
    return null;
  }

  /**
   * @return the response timeout at the connection provider to be applied in every request. This allows implementations to decide
   *         whether to expose this as a parameter or not in the connection provider. If empty is returned the default from Mule
   *         HttpClient will be used.
   */
  protected Optional<Integer> getResponseTimeout() {
    return empty();
  }

  /**
   * @return the definition for follows redirect at the connection provider to be applied in every request. This allows
   *         implementations to decide whether to expose this as a parameter or not in the connection provider. If empty is
   *         returned the default from Mule HttpClient will be used.
   */
  protected Optional<Boolean> isFollowsRedirect() {
    return empty();
  }

  private HttpRequestOptions getHttpRequestOptions() {
    HttpRequestOptionsBuilder httpRequestOptionsBuilder = HttpRequestOptions.builder()
        .authentication(authentication);
    getResponseTimeout().ifPresent(responseTimeout -> httpRequestOptionsBuilder.responseTimeout(responseTimeout));
    isFollowsRedirect().ifPresent(followsRedirect -> httpRequestOptionsBuilder.followsRedirect(followsRedirect));
    return httpRequestOptionsBuilder.build();
  }

}
