/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static com.mulesoft.connectivity.rest.commons.internal.util.MediaTypeUtils.getMediaType;
import static com.mulesoft.connectivity.rest.commons.internal.util.MediaTypeUtils.resolveCharset;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestUtils.consumeStringAndClose;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy.PagingStrategy;

import java.nio.charset.Charset;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

/**
 * Provides the base functionality for paginating the results of a REST endpoint.
 * <p>
 * Different APIs use different paging strategies which are tackled in implementations of this class. This class provides the
 * basics as:
 *
 * <ul>
 * <li>Executing one request per page to be obtained</li>
 * <li>Ability to parse the response to extract the data and useful metadata as the total results count (if provided)</li>
 * <li>Handling streaming</li>
 * <li>Handling the nuances of the {@link PagingProvider} contract</li>
 * <li>Handling encodings and media types</li>
 * </ul>
 * <p>
 * Each item in the obtained page will be returned as an individual {@link TypedValue}, with the proper mime type and encoding
 * set.
 * <p>
 * Type parameters: <C> – subclass of {@link RestConnection}
 * 
 * @since 1.0
 */
public abstract class RestPagingProvider<C extends RestConnection> implements PagingProvider<C, TypedValue<String>> {

  protected final MediaType defaultMediaType;

  private final Function<C, HttpRequestBuilder> requestFactory;

  private boolean stopPaging = false;

  private final PagingStrategy pagingStrategy;

  /**
   * Creates a new instance
   *
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance.
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one.
   * @param pagingStrategy the pagination strategy.
   */
  public RestPagingProvider(Function<C, HttpRequestBuilder> requestFactory,
                            MediaType defaultMediaType,
                            PagingStrategy pagingStrategy) {

    this.requestFactory = requestFactory;
    this.defaultMediaType = defaultMediaType;
    this.pagingStrategy = pagingStrategy;
  }

  @Override
  public final List<TypedValue<String>> getPage(C connection) {
    if (stopPaging) {
      return emptyList();
    }

    HttpRequestBuilder requestBuilder = requestFactory.apply(connection);
    requestBuilder = pagingStrategy.configureRequest(connection.getBaseUri(), requestBuilder, () -> stopPaging());

    if (stopPaging) {
      return emptyList();
    }

    // Send the REQUEST
    HttpResponse response = send(connection, requestBuilder);

    // Resolves MediaType and Charsets to consume response as String
    MediaType mediaType = getMediaType(response, defaultMediaType);
    Charset charset = resolveCharset(mediaType, defaultMediaType);
    TypedValue<String> rawPage = consumeStringAndClose(response.getEntity().getContent(),
                                                       mediaType,
                                                       charset);

    if (isBlank(rawPage.getValue()) || stopPaging) {
      return emptyList();
    }

    // Post processes response and extract pages
    List<TypedValue<String>> page = extractPageItems(postProcess(rawPage, defaultMediaType), defaultMediaType);

    // Calls the onPage hook to do something on every page according to the pagination strategy
    pagingStrategy.onPage(page, rawPage, response.getHeaders(), () -> stopPaging());

    return page;
  }

  /**
   * Does the HTTP call with the given connection for the requestBuilder.
   * 
   * @param connection raw content page as {@link TypedValue<String>}
   * @param requestBuilder the {@link HttpRequestBuilder} to be executed.
   * @return a {@link HttpResponse} which represents the server response.
   */
  protected HttpResponse send(C connection, HttpRequestBuilder requestBuilder) {
    try {
      return connection.send(requestBuilder.build());
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage(e.getMessage()), e);
    }
  }

  /**
   * Given the raw content response body this method will extract the payload data as a list of items which represent a page.
   * 
   * @param content raw content page as {@link TypedValue<String>}
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one.
   * @return a {@link List<TypedValue<String>>} which represents a page with items.
   */
  protected abstract List<TypedValue<String>> extractPageItems(TypedValue<String> content, MediaType defaultMediaType);

  /**
   * Allows post process the server response before the payload is extracted.
   *
   * @param rawPage the server response as it.
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one.
   * @return either the same rawPage or a new page modified.
   */
  protected TypedValue<String> postProcess(TypedValue<String> rawPage, MediaType defaultMediaType) {
    return rawPage;
  }


  @Override
  public Optional<Integer> getTotalResults(RestConnection connection) {
    return empty();
  }

  @Override
  public final void close(RestConnection connection) throws MuleException {}

  /**
   * Context related to the pagination.
   */
  public interface PagingContext {

    /**
     * Allows the strategy to stop the pagination mechanism.
     */
    void stopPaging();

  }

  private void stopPaging() {
    stopPaging = true;
  }

}
