/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

public interface RestConnection {

  /**
   * @return the base uri of the REST API being consumed.
   */
  String getBaseUri();

  /**
   * Asynchronous invocation of the {@link HttpRequest} against this connection. Allows non-blocking operations to fulfil the SDK
   * contract for non-blocking support in connectors.
   *
   * @param request to be executed.
   * @return {@link CompletableFuture<HttpResponse>} to handle the async response.
   */
  CompletableFuture<HttpResponse> sendAsync(HttpRequest request);

  /**
   * Synchronous invocation of the {@link HttpRequest} against the connection. Used for design operations like value providers or
   * even for paging operations where non-blocking is not supported by SDK.
   *
   * @param request to be executed.
   * @return {@link HttpResponse} the server response.
   * @throws IOException
   * @throws TimeoutException
   */
  HttpResponse send(HttpRequest request) throws IOException, TimeoutException;

  /**
   * Executes the {@link HttpRequest} and validates the response to generate the corresponding {@link ConnectionValidationResult}.
   *
   * @param request to be executed.
   * @param whenComplete to process an {@link HttpResponse} and return a {@link ConnectionValidationResult}.
   * @param onError to process an {@link Exception}, that happened from client side before calling the server, and return a
   *        {@link ConnectionValidationResult}.
   * @return a {@link ConnectionValidationResult}.
   */
  ConnectionValidationResult validate(HttpRequest request,
                                      Function<HttpResponse, ConnectionValidationResult> whenComplete,
                                      Function<Exception, ConnectionValidationResult> onError);

}
