/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation;

import static com.mulesoft.connectivity.rest.commons.api.error.RestError.CONNECTIVITY;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.TIMEOUT;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.getErrorTypeDefinitionByStatusCode;
import static java.lang.String.format;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.config.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.internal.util.MediaTypeUtils;

import java.util.Optional;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

import javax.inject.Inject;

public abstract class BaseRestOperation {

  @Inject
  ExpressionLanguage expressionLanguage;

  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  protected void setExpressionLanguage(ExpressionLanguage expressionLanguage) {
    checkArgument(expressionLanguage != null, "ExpressionLanguage cannot be null");
    this.expressionLanguage = expressionLanguage;
  }

  /**
   * Utility method that provides a {@link Function<HttpResponse, HttpResponse>} responsible for checking the {@link HttpResponse}
   * and based on {@link #httpResponseToErrorTypeDefinition()} the error type definition function defines if the response should
   * be treated as error or not. <br/>
   * <b>Be aware</b> that operations that make usage of this logic to throw errors should be annotated with Mule's SDK:
   * {@code @Throws(RestfullErrorTypeProvider.class)} if the {@link #httpResponseToErrorTypeDefinition()} has not been overridden
   * to change the {@link org.mule.sdk.api.annotation.error.ErrorTypeProvider}. <br/>
   * Example of usage in case of blocking operations:
   * 
   * <pre>
   * HttpResponse httpResponse = connection.send(HttpRequest.builder()
   *                          .uri(connection.getBaseUri() + "/accounts/" + accountId)
   *                          .build());
   * // Check if the HTTP response is not success based on RestfullError and throws a ModuleException
   * httpResponse = throwModuleExceptionIfErrorResponse().apply(httpResponse);
   * ...
   * </pre>
   * 
   * <br/>
   * Example of usage in case of non-blocking operations:
   * 
   * <pre>
   *  connection.sendAsync(HttpRequest.builder()
   *                          .uri(connection.getBaseUri() + "/accounts/" + accountId)
   *                          .build())
   *  // Check if the HTTP response is not success based on RestfullError and throws a ModuleException.
   * .thenApply(throwModuleExceptionIfErrorResponse())
   * .thenAccept(httpResponse ->...)
   * </pre>
   *
   * @return a {@link Function} to be applied after obtaining the {@link java.util.concurrent.CompletableFuture} from the
   *         {@link com.mulesoft.connectivity.rest.commons.api.connection.RestConnection} as it applies the decision logic to
   *         define if the {@link HttpResponse} is a failure or success response. If it is defined as an error a
   *         {@link ModuleException} is thrown with its associated {@link ErrorTypeDefinition}.
   */
  protected final Function<HttpResponse, HttpResponse> throwModuleExceptionIfErrorResponse() {
    return httpResponse -> {
      Optional<ErrorTypeDefinition<? extends Enum<?>>> errorTypeDefinitionOptional = httpResponseToErrorTypeDefinition()
          .apply(httpResponse);
      if (errorTypeDefinitionOptional.isPresent()) {
        ErrorTypeDefinition<?> errorTypeDefinition = errorTypeDefinitionOptional.get();
        throw createModuleException(httpResponse, errorTypeDefinition);
      }
      return httpResponse;
    };
  }

  /**
   * Template method that allows usage of custom {@link ErrorTypeDefinition}'s for cause where SaaS is not Restful complaint or
   * signal success error responses are returned. By overriding this method in a base abstract class in all the operations it will
   * allow customizing {@link ErrorTypeDefinition}'s. If the function returns an {@link Optional#empty()} it means the
   * {@link HttpResponse} is a success. If this method is overridden most like {@link #toConnectivityErrorFunction()} would need
   * to be too as it uses {@link RestError} for generic errors. <br/>
   * For scenarios like signal error responses with success status code responses it is recommended to materialize the
   * {@link HttpResponse} entity content before calling {@link #throwModuleExceptionIfErrorResponse()} with one that loads in
   * memory the response with an {@link org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity} so it could be consumed more
   * than once.
   *
   * @return a {@link Function<HttpResponse, Optional<ErrorTypeDefinition<? extends Enum<?>>>} to check if the response
   *         corresponds to an error.
   */
  protected Function<HttpResponse, Optional<ErrorTypeDefinition<? extends Enum<?>>>> httpResponseToErrorTypeDefinition() {
    return httpResponse -> getErrorTypeDefinitionByStatusCode(httpResponse.getStatusCode());
  }

  /**
   * Utility method that provides a hook for customizing the {@link ModuleException} to be thrown when an {@link HttpResponse} is
   * marked as failure based on {@link #httpResponseToErrorTypeDefinition()}. <br/>
   * Default implementation will generate an {@code error.description} based on {@code statusCode " " reasonPhrase}.
   *
   * @param httpResponse the response {@link HttpResponse} from the server.
   * @param errorTypeDefinition {@link ErrorTypeDefinition} associated to this failure response.
   * @param <E> ErrorType generic.
   * @return a {@link ModuleException} with the description and the {@link ErrorTypeDefinition} associated.
   */
  protected <E extends Enum<E>> ModuleException createModuleException(HttpResponse httpResponse,
                                                                      ErrorTypeDefinition<E> errorTypeDefinition) {
    return new ModuleException(httpResponse.getStatusCode() + " " + httpResponse.getReasonPhrase(),
                               errorTypeDefinition);
  }

  /**
   * Utility method that provides a {@link Function<HttpResponse, HttpResponse>} responsible for handling {@link Throwable}s and
   * notify the {@link CompletionCallback} on non-blocking operations only. It will unwrap a {@link CompletionException} to get
   * the cause and convert to {@link ModuleException} with its corresponding {@link ErrorTypeDefinition}. If the exception is
   * already a {@link ModuleException} it will be propagated as it may have been thrown by
   * {@link #throwModuleExceptionIfErrorResponse()}. <br/>
   * Example of usage in a non-blocking operation:
   * 
   * <pre>
   * connection.sendAsync(HttpRequest.builder()
   *     .uri(connection.getBaseUri() + "/accounts/" + accountId)
   *     .build())
   *     // Check if the HTTP response is not success based on RestfullError and throws a ModuleException.
   *     .thenApply(throwModuleExceptionIfErrorResponse())
   *     .thenAccept(httpResponse -> completionCallback.success(Result.<String, Integer>builder()
   *         .output(IOUtils.toString(httpResponse.getEntity().getContent()))
   *         .attributes(httpResponse.getStatusCode())
   *         .attributesMediaType(APPLICATION_JAVA)
   *         .build()))
   *     // Converts to a ModuleException with the default generic ErrorType: CONNECTIVITY or TIMEOUT from RestfullError
   *     // and notifies the completionCallback with the error.
   *     .exceptionally(notifyCompletionCallbackError(completionCallback));
   * </pre>
   *
   * @param completionCallback the {@link CompletionCallback} to be notified about the error.
   * @param <P> type for output.
   * @param <A> type for output attributes.
   * @return a {@link Function} to be provided as {@link java.util.concurrent.CompletableFuture#exceptionally(Function)} to handle
   *         any {@link Throwable} that happens during the execution of the completable future. Its responsible for notifying the
   *         {@link CompletionCallback} and translated to {@link ModuleException} the exception with its corresponding
   *         {@link ErrorTypeDefinition}.
   */
  protected final <P, A, T> Function<Throwable, T> notifyCompletionCallbackError(CompletionCallback<P, A> completionCallback) {
    return throwable -> {
      Throwable rootCause = throwable;
      if (throwable instanceof CompletionException) {
        rootCause = throwable.getCause();
      }
      completionCallback.error(toConnectivityErrorFunction().apply(rootCause));
      return null;
    };
  }

  /**
   * Template method that allows usage of custom {@link ErrorTypeDefinition}'s for general errors like {@link TimeoutException} or
   * a generic error such as {@link RestError#CONNECTIVITY}. The implementation should consider that a {@link ModuleException} may
   * be passed to this function in which case it should be propagated as it is. By overriding this method in a base abstract class
   * in all the operations it will allow customizing these {@link ErrorTypeDefinition}'s. If these generic errors are modified the
   * operation should be consistent about the {@link org.mule.runtime.extension.api.annotation.error.ErrorTypeProvider} and most
   * likely {@link #httpResponseToErrorTypeDefinition()} would need to be overridden too.
   *
   * @return a {@link Function<Throwable, ModuleException>} to handle a {@link Throwable} and return a {@link ModuleException}
   *         with its corresponding {@link ErrorTypeDefinition}.
   */
  protected Function<Throwable, ModuleException> toConnectivityErrorFunction() {
    return throwable -> {
      if (throwable instanceof ModuleException) {
        return (ModuleException) throwable;
      }
      if (throwable instanceof TimeoutException) {
        return new ModuleException(TIMEOUT, throwable);
      }
      return new ModuleException(CONNECTIVITY, throwable);
    };
  }

  /**
   * This method specifies the {@link MediaType} that should be assumed the response to have in case the remote service doesn't
   * specify a {@code Content-Type} header. <br/>
   * By default, this method returns {@link MediaType#APPLICATION_JSON} since it's the most common response type. However, this
   * method should be overwritten if a different type of media type is to be expected, or if you know that a certain encoding will
   * be enforced.
   *
   * @return the {@link MediaType} to assign the response in case the server doesn't specify one in its response
   */
  protected MediaType getDefaultResponseMediaType() {
    return MediaType.APPLICATION_JSON;
  }

  /**
   * Resolves the default {@link MediaType} to be used when processing the response.
   * 
   * @param config {@link RestConfiguration} to get the defaultCharset from Mule Runtime.
   * @return the {@link MediaType}.
   */
  protected final MediaType resolveDefaultResponseMediaType(RestConfiguration config) {
    MediaType mediaType = getDefaultResponseMediaType();
    if (!mediaType.getCharset().isPresent()) {
      mediaType = mediaType.withCharset(config.getCharset());
    }

    return mediaType;
  }

  /**
   * Given the default response media type obtained by {@link #resolveDefaultResponseMediaType(RestConfiguration)} and the
   * {@link HttpResponse} it will get the {@link MediaType} from the
   * {@link org.mule.runtime.http.api.HttpHeaders.Names#CONTENT_TYPE} or use the {@code defaultResponseMediaType} provided as
   * parameter (which should be obtained from {@link #resolveDefaultResponseMediaType(RestConfiguration)}. <br/>
   * 
   * <pre>
   * connection.sendAsync(HttpRequest.builder()
   *     .uri(connection.getBaseUri() + "/accounts/" + accountId)
   *     .build())
   *     .thenApply(throwModuleExceptionIfErrorResponse())
   *     .thenAccept(httpResponse -> completionCallback.success(Result.<InputStream, Void>builder()
   *         .output(httpResponse.getEntity().getContent())
   *         .mediaType(getMediaType(httpResponse, resolveDefaultResponseMediaType(configuration)))
   *         .build()))
   *     .exceptionally(notifyCompletionCallbackError(completionCallback));
   * </pre>
   *
   * @param response the {@link HttpResponse}
   * @param defaultResponseMediaType to be used if header is missed.
   * @return {@link MediaType} to be set to the
   *         {@link org.mule.runtime.extension.api.runtime.operation.Result.Builder#mediaType(MediaType)}.
   */
  protected final MediaType getMediaType(HttpResponse response, MediaType defaultResponseMediaType) {
    return MediaTypeUtils.getMediaType(response, defaultResponseMediaType);
  }

}
