/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy;

import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import com.mulesoft.connectivity.rest.commons.api.operation.paging.RestPagingProvider.PagingContext;

import java.util.List;

/**
 * Base class of {@link PagingStrategy} for APIs which do paging based on a marker provided in the response of each request,
 * acting as a marker or position inside the stream of results.
 * <p>
 * Request number 1 will obtain the first page plus a marker embedded in the response. Said marker will be added as a query param
 * in request number 2, which will reply with the next page and a new marker which overwrites the prior one. The process continues
 * until no new marker is provided or the next page comes back empty.
 *
 * @since 1.0
 */
public abstract class MarkerPagingStrategy implements PagingStrategy {

  private final String markerParamName;

  private boolean firstPage = true;
  private String nextMarker;

  /**
   * Creates a new instance
   *
   * @param markerParamName the name of the query parameter in which the marker is placed in the request
   */
  public MarkerPagingStrategy(String markerParamName) {
    this.markerParamName = markerParamName;
  }

  @Override
  public HttpRequestBuilder configureRequest(String baseUri, HttpRequestBuilder requestBuilder, PagingContext context) {
    if (!firstPage) {
      if (isBlank(nextMarker)) {
        context.stopPaging();
        return requestBuilder;
      }

      requestBuilder.addQueryParam(markerParamName, nextMarker);
    }
    return requestBuilder;
  }

  @Override
  public void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage, MultiMap<String, String> headers,
                     PagingContext context) {
    firstPage = false;

    if (isBlank(rawPage.getValue())) {
      nextMarker = null;
      context.stopPaging();
    } else {
      nextMarker = getNextMarker(rawPage, headers);
    }
  }

  protected abstract String getNextMarker(TypedValue<String> rawPage, MultiMap<String, String> headers);

}
