/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.parse;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.ErrorMessageAwareException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.dw.CommonsExpressionLanguage;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseResult;
import com.mulesoft.connectivity.rest.commons.internal.dw.CommonsExpressionLanguageImpl;
import com.mulesoft.connectivity.rest.commons.internal.http.HttpResponseAttributes;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

import org.slf4j.Logger;

/**
 * Base class for implementing {@link RestConnection} which provides an implementation based on {@link HttpClient} and supports
 * for error handling which allow to be customized by sub classing with template methods.
 *
 * @param <T> defines the ErrorType enumeration.
 */
public abstract class BaseRestConnection<E extends ModuleException, T extends ErrorTypeDefinition<? extends Enum<?>>>
    implements RestConnection {

  private static final Logger LOGGER = getLogger(BaseRestConnection.class);

  private final HttpClient httpClient;
  private final HttpRequestOptions httpRequestOptions;
  private final String baseUri;
  private ExpressionLanguage expressionLanguage;

  public BaseRestConnection(HttpClient httpClient, HttpRequestOptions httpRequestOptions, String baseUri,
                            ExpressionLanguage expressionLanguage) {
    requireNonNull(httpClient, "httpClient cannot be null");
    requireNonNull(httpRequestOptions, "httpRequestOptions cannot be null");
    requireNonNull(expressionLanguage, "expressionLanguage cannot be null");

    this.httpClient = httpClient;
    this.httpRequestOptions = httpRequestOptions;
    this.baseUri = baseUri;
    this.expressionLanguage = expressionLanguage;
  }

  protected HttpClient getHttpClient() {
    return httpClient;
  }

  protected HttpRequestOptions getHttpRequestOptions() {
    return httpRequestOptions;
  }

  public String getBaseUri() {
    return baseUri;
  }

  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  protected CommonsExpressionLanguage getCommonsExpressionLanguage() {
    return new CommonsExpressionLanguageImpl(expressionLanguage);
  }

  @Override
  public CompletableFuture<HttpResponseResult> sendAsync(HttpRequest request, MediaType defaultResponseMediaType) {
    CompletableFuture<HttpResponseResult> future = new CompletableFuture<>();
    try {
      getHttpClient().sendAsync(beforeRequest(request), getHttpRequestOptions())
          .whenComplete((httpResponse, throwable) -> {
            if (throwable != null) {
              // The throwable would be a CompletionException, so we get its cause in order to convert to
              // ModuleException
              future.completeExceptionally(toConnectivityErrorFunction(throwable.getCause()));
              return;
            }

            MediaType mediaType = getMediaType(httpResponse, defaultResponseMediaType);
            try {
              HttpResponse response = throwModuleExceptionIfErrorResponse(httpResponse, mediaType);
              future.complete(toResult(response, mediaType));
            } catch (Exception e) {
              future.completeExceptionally(toConnectivityErrorFunction(e));
            }
          });
    } catch (Throwable t) {
      future.completeExceptionally(toConnectivityErrorFunction(t));
    }
    return future;
  }

  protected final MediaType getMediaType(HttpResponse response, MediaType defaultResponseMediaType) {
    checkArgument(defaultResponseMediaType.getCharset().isPresent(), "'defaultResponseMediaType' should have a charset defined");

    MediaType contentType = defaultResponseMediaType;
    String responseContentType = response.getHeaders().get(CONTENT_TYPE);
    if (responseContentType != null) {
      try {
        contentType = parse(responseContentType);
      } catch (Exception e) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(format("Server response 'Content-Type': '%s' could not be parsed to a valid Media Type. Will ignore",
                              responseContentType),
                       e);
        }
        return contentType;
      }
      if (!contentType.getCharset().isPresent()) {
        if (!contentType.matches(defaultResponseMediaType)) {
          throw new MuleRuntimeException(
                                         createStaticMessage("Server response 'Content-Type': '%s' is missing charset but 'defaultResponseMediaType': '%s' doesn't match to resolve the charset/encoding",
                                                             contentType, defaultResponseMediaType));
        } else {
          contentType = contentType.withCharset(defaultResponseMediaType.getCharset().get());
        }
      }
    }
    return contentType;
  }

  protected HttpResponseResult toResult(HttpResponse httpResponse, MediaType mediaType) {
    HttpEntity entity = httpResponse.getEntity();
    return HttpResponseResult.builder()
        .entityContent(entity.getContent())
        .mediaType(mediaType)
        .length(entity.getBytesLength())
        .httpResponseAttributes(new HttpResponseAttributes(httpResponse.getStatusCode(), httpResponse.getReasonPhrase(),
                                                           httpResponse.getHeaders()))
        .build();
  }

  @Override
  public HttpResponseResult send(HttpRequest request, MediaType defaultResponseMediaType) throws E {
    try {
      HttpResponse httpResponse = getHttpClient().send(beforeRequest(request), getHttpRequestOptions());
      MediaType mediaType = getMediaType(httpResponse, defaultResponseMediaType);
      return toResult(throwModuleExceptionIfErrorResponse(httpResponse, mediaType), mediaType);
    } catch (Exception e) {
      throw toConnectivityErrorFunction(e);
    }
  }

  @Override
  public ConnectionValidationResult validate(HttpRequest request,
                                             Function<HttpResponse, ConnectionValidationResult> whenComplete,
                                             Function<Exception, ConnectionValidationResult> onError) {
    try {
      return whenComplete.apply(getHttpClient().send(beforeRequest(request), getHttpRequestOptions()));
    } catch (Exception e) {
      return onError.apply(e);
    }
  }

  protected HttpRequest beforeRequest(HttpRequest request) {
    HttpRequestBuilder httpRequestBuilder = HttpRequest.builder()
        .method(request.getMethod())
        .protocol(request.getProtocol())
        .uri(request.getUri())
        .queryParams(request.getQueryParams())
        .headers(request.getHeaders())
        .entity(request.getEntity());

    authenticate(httpRequestBuilder);

    return httpRequestBuilder.build();
  }

  /**
   * Allows to different type of connections to intercept the request before sending the HTTP call. This would be useful for
   * Bearer, API Key or OAuth connections which need to include headers for authentication.
   *
   * @param httpRequestBuilder {@link HttpRequestBuilder} populated with the values from the original {@link HttpRequest}.
   * @return either the same request or one modified.
   */
  protected void authenticate(HttpRequestBuilder httpRequestBuilder) {
    // Nothing to do by default
  }

  /**
   * Utility method that provides a {@link Function<HttpResponse, HttpResponse>} responsible for checking if the
   * {@link HttpResponse} should be treated as error, based on {@link #httpResponseToErrorTypeDefinition(HttpResponse, MediaType)}
   * error type definition function. <br/>
   * <b>Be aware</b> that operations that make usage of this connection which throws errors should be annotated with Mule's SDK:
   * {@code @Throws(RestfullErrorTypeProvider.class)}, if the {@link #httpResponseToErrorTypeDefinition(HttpResponse, MediaType)}
   * has not been overridden to change the {@link org.mule.runtime.extension.api.annotation.error.ErrorTypeProvider}. <br/>
   * Example of usage in case of blocking requests:
   *
   * <pre>
   * HttpResponse httpResponse = httpClient.send(HttpRequest.builder()
   *                          .uri(connection.getBaseUri() + "/accounts/" + accountId)
   *                          .build());
   * // Check if the HTTP response is not success based on RestfullError and throws a ModuleException
   * httpResponse = throwModuleExceptionIfErrorResponse().apply(httpResponse);
   * ...
   * </pre>
   *
   * <br/>
   * Example of usage in case of non-blocking requests:
   *
   * <pre>
   *  httpClient.sendAsync(HttpRequest.builder()
   *                          .uri(connection.getBaseUri() + "/accounts/" + accountId)
   *                          .build())
   *  // Check if the HTTP response is not success based on RestfullError and throws a ModuleException.
   * .thenApply(throwModuleExceptionIfErrorResponse())
   * .thenAccept(httpResponse ->...)
   * </pre>
   *
   * @param mediaType to read the entity content from server.
   * @return a {@link Function} to be applied after obtaining the {@link java.util.concurrent.CompletableFuture} from the
   *         {@link HttpClient} invocation as it applies the decision logic to define if the {@link HttpResponse} is a failure or
   *         success response. If it is defined as an error a {@link ModuleException} is thrown with its associated
   *         {@link ErrorTypeDefinition}.
   */
  protected final HttpResponse throwModuleExceptionIfErrorResponse(HttpResponse httpResponse, MediaType mediaType) {
    httpResponse = HttpResponse.builder()
        .headers(httpResponse.getHeaders())
        .statusCode(httpResponse.getStatusCode())
        .reasonPhrase(httpResponse.getReasonPhrase())
        .entity(materializeHttpResponseEntity(httpResponse.getEntity()))
        .build();
    Optional<T> errorTypeDefinitionOptional = httpResponseToErrorTypeDefinition(httpResponse, mediaType);
    if (errorTypeDefinitionOptional.isPresent()) {
      T errorTypeDefinition = errorTypeDefinitionOptional.get();
      throw createModuleException(httpResponse, errorTypeDefinition, mediaType);
    }
    return httpResponse;
  }

  /**
   * Whenever the connection implementation needs to handle signal error responses which are 200 response status code the method
   * {@link #httpResponseToErrorTypeDefinition(HttpResponse, MediaType)} would be implemented in order to read the entity content
   * response to check for the signal error flag. If this is done without materializing the HttpEntity it won't later allow the
   * client which made the request to be executed by the connection to consume the response. Therefore, this method allows to
   * materialize the entity so it would allow to be read more than once.
   *
   * @param httpEntity the server's response.
   * @return an {@link HttpEntity} that allows to be read more than once.
   */
  protected abstract HttpEntity materializeHttpResponseEntity(HttpEntity httpEntity);

  /**
   * Template method that allows usage of custom {@link ErrorTypeDefinition}'s for those scenarios where SaaS is not Restful
   * complaint or signal success error responses are returned with 20x responses. By overriding this method in a custom connection
   * implementation allows customizing {@link ErrorTypeDefinition}'s. If the function returns an {@link Optional#empty()} it means
   * the {@link HttpResponse} is a success. If this method is overridden most like {@link #toConnectivityErrorFunction(Throwable)}
   * would need to be too as it uses {@link RestError} for generic errors. <br/>
   * For scenarios like signal error responses with success status code responses it is recommended to materialize the
   * {@link HttpResponse} entity content with an in memory implementation such is
   * {@link org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity} by overwritten
   * {@link #materializeHttpResponseEntity(HttpEntity)} so it could be consumed more than once, to check the body content for
   * signal error and if it not an error by the client when processing the response.
   *
   * @param httpResponse the response {@link HttpResponse} from the server.
   * @param mediaType to read the entity content from server.
   * @return a {@link Optional<T>} to check if the response corresponds to an error.
   */
  protected abstract Optional<T> httpResponseToErrorTypeDefinition(HttpResponse httpResponse, MediaType mediaType);

  /**
   * Utility method that provides a hook for customizing the {@link E} to be thrown when an {@link HttpResponse} is marked as
   * failure based on {@link #httpResponseToErrorTypeDefinition(HttpResponse, MediaType)}. <br/>
   * Default implementation will generate an {@code error.description} based on {@code statusCode " " reasonPhrase}.
   *
   * <br/>
   * It is recommended to write in the connector's base code a specific {@link E} that also implements
   * {@link ErrorMessageAwareException} so it associates the {@link HttpResponse} body as {@code payload} and the
   * {@code attributes} representation's for the response.
   *
   * @param httpResponse the response {@link HttpResponse} from the server.
   * @param errorTypeDefinition {@link ErrorTypeDefinition} associated to this failure response.
   * @param mediaType to read the entity content from server.
   * @return a {@link E} with the description and the {@link ErrorTypeDefinition} associated.
   */
  protected abstract E createModuleException(HttpResponse httpResponse,
                                             T errorTypeDefinition,
                                             MediaType mediaType);

  /**
   * Template method that allows usage of custom {@link ErrorTypeDefinition}'s for general errors like {@link TimeoutException} or
   * {@link java.io.IOException} which are thrown by the {@link HttpClient} API. The implementation should consider that a
   * {@link E} may be passed to this function in which case it should be propagated as it is. By overriding this method in a
   * custom connection implementation it will allow customizing these {@link ErrorTypeDefinition}'s.
   *
   * @return a {@link Function<Throwable, E>} to handle a {@link Throwable} and return a {@link E} with its corresponding
   *         {@link ErrorTypeDefinition}.
   */
  protected abstract E toConnectivityErrorFunction(Throwable throwable);
}
