/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseResult;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

/**
 * A connection is responsible for executing the {@link HttpRequest} to the server and generate the {@link HttpResponseResult
 * <InputStream, HttpResponseAttributes>} for the response with the body and HTTP protocol attributes. It also validates if the
 * server is valid. It handles responses according to the error handling definition, if a request generates a failure response it
 * will throw a {@link ModuleException} with the specific {@link org.mule.runtime.api.message.ErrorType} defined for it.
 *
 * @since 1.0
 */
public interface RestConnection {

  /**
   * @return the base uri of the REST API being consumed. Could be null.
   */
  String getBaseUri();

  /**
   * Asynchronous invocation of the {@link HttpRequest} against this connection. Allows non-blocking operations to fulfil the SDK
   * contract for non-blocking support in connectors.
   *
   * @param request to be executed.
   * @param defaultResponseMediaType to be used if the server doesn't specify one with the
   *        {@link org.mule.runtime.http.api.HttpHeaders.Names#CONTENT_TYPE} header.
   * @return {@link CompletableFuture< HttpResponseResult >} to handle the async response. CompletableFuture will success if the
   *         response from the server is success, case of a failure a {@link ModuleException} would be passed to the
   *         completeExceptionally. The {@link HttpResponseResult} provides the {@link MediaType} for the output
   *         {@link InputStream}.
   */
  CompletableFuture<HttpResponseResult> sendAsync(HttpRequest request, MediaType defaultResponseMediaType);

  /**
   * Synchronous invocation of the {@link HttpRequest} against the connection. Used for design operations like value providers or
   * even for paging operations where non-blocking is not supported by SDK.
   *
   * @param request to be executed.
   * @param defaultResponseMediaType to be used if the server doesn't specify one with the
   *        {@link org.mule.runtime.http.api.HttpHeaders.Names#CONTENT_TYPE} header.
   * @return {@link HttpResponseResult <InputStream, HttpResponseAttributes>} with the response body and HTTP protocol attributes
   *         from the server. The {@link HttpResponseResult} provides the {@link MediaType} for the output {@link InputStream}.
   * @throws {@link ModuleException} in case if the response is a failure, a timeout or an error while trying to reach out the
   *         server happened when sending the request.
   */
  HttpResponseResult send(HttpRequest request, MediaType defaultResponseMediaType) throws ModuleException;

  /**
   * Executes the {@link HttpRequest} and validates the response to generate the corresponding {@link ConnectionValidationResult}.
   *
   * @param request to be executed.
   * @param whenComplete to process an {@link HttpResponse} and return a {@link ConnectionValidationResult}.
   * @param onError to process an {@link Exception}, that happened from client side before calling the server, and return a
   *        {@link ConnectionValidationResult}.
   * @return a {@link ConnectionValidationResult}.
   */
  ConnectionValidationResult validate(HttpRequest request,
                                      Function<HttpResponse, ConnectionValidationResult> whenComplete,
                                      Function<Exception, ConnectionValidationResult> onError);

}
