/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection.oauth;

import static com.mulesoft.connectivity.rest.commons.api.error.RestError.UNAUTHORIZED;
import static java.util.Objects.requireNonNull;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.connectivity.oauth.AccessTokenExpiredException;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthState;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.connection.DefaultRestConnection;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;

/**
 * Specialization of {@link DefaultRestConnection} for resources protected with OAuth, regardless of the grant type.
 */
public class OAuthRestConnection<O extends OAuthState> extends DefaultRestConnection {

  private final O oauthState;
  private final String resourceOwnerId;

  public OAuthRestConnection(HttpClient httpClient,
                             HttpRequestOptions httpRequestOptions, String baseUri,
                             ExpressionLanguage expressionLanguage, O oauthState, String resourceOwnerId) {
    super(httpClient, assertNoAuthenticationProvided(httpRequestOptions), baseUri, expressionLanguage);
    requireNonNull(oauthState, "oauthState cannot be null");

    this.oauthState = oauthState;
    this.resourceOwnerId = resourceOwnerId;
  }

  private static HttpRequestOptions assertNoAuthenticationProvided(HttpRequestOptions httpRequestOptions) {
    if (httpRequestOptions.getAuthentication().isPresent()) {
      throw new IllegalStateException("OAuthRestConnection should not be created with an HttpRequestOptions that provides an Authentication");
    }
    return httpRequestOptions;
  }

  @Override
  protected final void authenticate(HttpRequestBuilder httpRequestBuilder) {
    authenticate(httpRequestBuilder, oauthState);
  }

  /**
   * Allows to implement a custom signature for OAuth.
   * 
   * @param httpRequestBuilder {@link HttpRequestBuilder} to sign the request.
   * @param oauthState {@link OAuthState} to get access to the access token.
   */
  protected void authenticate(HttpRequestBuilder httpRequestBuilder, O oauthState) {
    httpRequestBuilder.addHeader(AUTHORIZATION, "Bearer " + oauthState.getAccessToken());
  }

  @Override
  protected ModuleException toConnectivityErrorFunction(Throwable throwable) {
    ModuleException moduleException = super.toConnectivityErrorFunction(throwable);
    try {
      throwAccessTokenExpiredExceptionIfNeedsRefreshToken(moduleException.getType(), resourceOwnerId);
    } catch (AccessTokenExpiredException e) {
      return new ModuleException(moduleException.getType(), e);
    }
    return moduleException;
  }

  /**
   * Template method to allow custom code for triggering a refresh token in case of a different error or custom error types.
   * Default implementation is based on {@link RestError#UNAUTHORIZED} error code throw the {@link AccessTokenExpiredException} so
   * Mule Runtime/SDK will refresh the access token. <br/>
   * For scenarios where SaaS/Server response is complaint with statusCode and uses a signal error in the body response to trigger
   * the refresh token this method may not need to be implemented (it depends if the error type definition needs to be changed)
   * but instead the following methods need to be implemented
   * {@link DefaultRestConnection#materializeHttpResponseEntity(HttpEntity)} in order to allow reading the response entity content
   * more than once and {@link DefaultRestConnection#httpResponseToErrorTypeDefinition(HttpResponse, MediaType)} to associate the
   * response with a {@link RestError#UNAUTHORIZED} or the one that represents this in the custom error type definition for the
   * connector.
   *
   * @param errorTypeDefinition associated to the {@link HttpResponse} for the request executed.
   * @param resourceOwnerId the oauth resource owner id to be provided by the {@link AccessTokenExpiredException} or null.
   * @return a {@link HttpResponse} to be returned by the connection to the caller.
   */
  protected void throwAccessTokenExpiredExceptionIfNeedsRefreshToken(ErrorTypeDefinition errorTypeDefinition,
                                                                     String resourceOwnerId) {
    if (UNAUTHORIZED.name() == errorTypeDefinition.getType()) {
      if (resourceOwnerId == null) {
        throw new AccessTokenExpiredException();
      }
      throw new AccessTokenExpiredException(resourceOwnerId);
    }
  }

}
