/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.dw;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.metadata.MediaType;

import com.mulesoft.connectivity.rest.commons.internal.dw.CommonsBindingContextBuilder;

import java.io.InputStream;
import java.nio.charset.Charset;

/**
 * Wrapping class of a {@link BindingContext} in which it's creation simplifies the mechanism to add bindings (removing the usages
 * of typed values). For its usage, see {@link CommonsExpressionLanguage}
 */
public interface CommonsBindingContext {

  /**
   * Provides a builder to create {@link CommonsBindingContext} objects.
   *
   * @return a new {@link CommonsBindingContext.Builder}.
   */
  static CommonsBindingContext.Builder builder() {
    return new CommonsBindingContextBuilder();
  }

  /**
   * @return a new {@link BindingContext}
   */
  BindingContext bindingContext();


  interface Builder {

    /**
     * Will create a binding for the specified identifier and value with the specified media type. <br/>
     * Use this when working with dynamic {@link MediaType} and it is not known at the moment a transformation needs to be done.
     *
     * @param value the content value to bind
     * @param identifier the keyword to use in the EL to access the {@code value}
     * @param mediaType the media type of the InputStream
     */
    Builder addContent(String identifier, InputStream value, MediaType mediaType);

    /**
     * Will create a binding for the specified identifier and value of a JSON type with the specified encoding.
     *
     * @param value the JSON value to bind
     * @param identifier the keyword to use in the EL to access the {@code value}
     * @param charset the encoding of the InputStream
     */
    Builder addJson(String identifier, InputStream value, Charset charset);

    /**
     * Will create a binding for the specified identifier and value of a JSON type.
     *
     * @param value the JSON value to bind
     * @param identifier the keyword to use in the EL to access the {@code value}
     */
    Builder addJson(String identifier, CommonsExpressionLanguageValue value);

    /**
     * Will create a binding for the specified identifier and value of an XML type with the specified encoding.
     *
     * @param value the XML value to bind
     * @param identifier the keyword to use in the EL to access the {@code value}
     * @param charset the encoding of the InputStream
     */
    Builder addXml(String identifier, InputStream value, Charset charset);

    /**
     * Will create a binding for the specified identifier and value of an XML type.
     *
     * @param value the XML value to bind
     * @param identifier the keyword to use in the EL to access the {@code value}
     */
    Builder addXml(String identifier, CommonsExpressionLanguageValue value);

    /**
     * Will create a binding for the specified identifier and value of Java Object type.
     *
     * @param value the Object value to bind
     * @param identifier the keyword to use in the EL to access the {@code value}
     */
    Builder addJava(String identifier, Object value);

    /**
     * @return a new {@link CommonsBindingContext}
     */
    CommonsBindingContext build();
  }

}
