/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.dw;

import org.mule.runtime.api.el.ExpressionExecutionException;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.DataType;

/**
 * Evaluates an expression considering a set of given bindings and a set of global ones in a simplified way, for a broader usage
 * we recommend using {@link ExpressionLanguage}
 *
 * @since 1.0
 */
public interface CommonsExpressionLanguage {

  /**
   * Evaluates an expression according to a given {@link CommonsBindingContext} and the global one.
   *
   * @param expression the expression which expected output type must be a JSON
   * @param commonsBindingContext the current dynamic binding context to consider
   * @return a JSON {@link CommonsExpressionLanguageValue} result of the expression
   * @throws ExpressionExecutionException when an error occurs during evaluation
   */
  CommonsExpressionLanguageValue evaluateJson(String expression, CommonsBindingContext commonsBindingContext)
      throws ExpressionExecutionException;

  /**
   * Evaluates an expression according to a given {@link CommonsBindingContext} and the global one.
   *
   * @param expression the expression which expected output type must be an XML
   * @param commonsBindingContext the current dynamic binding context to consider
   * @return a XML {@link CommonsExpressionLanguageValue} result of the expression
   * @throws ExpressionExecutionException when an error occurs during evaluation
   */
  CommonsExpressionLanguageValue evaluateXml(String expression, CommonsBindingContext commonsBindingContext)
      throws ExpressionExecutionException;

  /**
   * Evaluates an expression according to a given {@link CommonsBindingContext} and the global one and a specific data type.
   *
   * @param expression the expression which expected output type must be the same as the dataType parameter
   * @param commonsBindingContext the current dynamic binding context to consider
   * @param dataType the result data type
   * @return a {@link CommonsExpressionLanguageValue} result of the expression
   * @throws ExpressionExecutionException when an error occurs during evaluation
   */
  CommonsExpressionLanguageValue evaluate(String expression, CommonsBindingContext commonsBindingContext, DataType dataType)
      throws ExpressionExecutionException;

  /**
   * Evaluates an expression according to a given {@link CommonsBindingContext} and the global one.
   *
   * @param expression the expression which expected output type must be a Java String
   * @param commonsBindingContext the current dynamic binding context to consider
   * @return a Java String result of the expression
   * @throws ExpressionExecutionException when an error occurs during evaluation
   */
  String evaluateAsJavaString(String expression, CommonsBindingContext commonsBindingContext) throws ExpressionExecutionException;

  /**
   * Evaluates an expression according to a given {@link CommonsBindingContext} and the global one.
   *
   * @param expression the expression which expected output type must be a Java Boolean
   * @param commonsBindingContext the current dynamic binding context to consider
   * @return a Java Boolean result of the expression
   * @throws ExpressionExecutionException when an error occurs during evaluation
   */
  Boolean evaluateAsJavaBoolean(String expression, CommonsBindingContext commonsBindingContext)
      throws ExpressionExecutionException;

  /**
   * Evaluates an expression according to a given {@link CommonsBindingContext} and the global one.
   *
   * @param expression the expression which expected output type must be a Java Number
   * @param commonsBindingContext the current dynamic binding context to consider
   * @return a Java Number result of the expression
   * @throws ExpressionExecutionException when an error occurs during evaluation
   */
  Number evaluateAsJavaNumber(String expression, CommonsBindingContext commonsBindingContext) throws ExpressionExecutionException;

}
