/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.error;

import static java.util.Optional.ofNullable;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.getStatusByCode;

import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.error.MuleErrors;
import org.mule.runtime.http.api.HttpConstants.HttpStatus;

import java.util.Optional;

/**
 * Represents an error that can happen in an HTTP operation. It should be used by connectors that connects to RESTful APIs where
 * error statusCodes are complaints with HTTP. Signal errors responses for SaaS that provide REST APIs where errors are signaled
 * in the body response and statusCodes is 200 (OK) should have their own errorTypeDefinition.
 *
 * @since 1.0
 */
public enum RestError implements ErrorTypeDefinition<RestError> {

  CONNECTIVITY(MuleErrors.CONNECTIVITY),

  CLIENT_ERROR(CONNECTIVITY), TIMEOUT(CLIENT_ERROR), UNAUTHORIZED(CLIENT_ERROR), NOT_FOUND(CLIENT_ERROR), TOO_MANY_REQUESTS(
      CLIENT_ERROR), BAD_REQUEST(CLIENT_ERROR), UNSUPPORTED_MEDIA_TYPE(CLIENT_ERROR),

  NOT_ACCEPTABLE(CONNECTIVITY),

  SERVER_ERROR(CONNECTIVITY), INTERNAL_SERVER_ERROR(SERVER_ERROR), SERVICE_UNAVAILABLE(SERVER_ERROR);

  private ErrorTypeDefinition<?> parentErrorType;
  private final String message;

  RestError() {
    this.message = this.name().replace("_", " ").toLowerCase();
  }

  RestError(ErrorTypeDefinition<?> parentErrorType) {
    this();
    this.parentErrorType = parentErrorType;
  }

  @Override
  public Optional<ErrorTypeDefinition<? extends Enum<?>>> getParent() {
    return ofNullable(parentErrorType);
  }

  /**
   * Returns the {@link RestError} corresponding to a given status code. A match is found if there's an {@link RestError} with the
   * same name as the status code's corresponding {@link HttpStatus}.
   *
   * @param statusCode the HTTP status code to search for
   * @return an {@link Optional} with the {@link RestError} that matches the {@code statusCode}
   */
  public static Optional<ErrorTypeDefinition<RestError>> getErrorTypeDefinitionByStatusCode(int statusCode) {
    RestError error = null;
    HttpStatus status = getStatusByCode(statusCode);
    if (status != null) {
      try {
        error = RestError.valueOf(status.name());
      } catch (IllegalArgumentException e) {
        // Do nothing. Will handle this in the next if block
      }
    }

    if (error == null) {
      if (statusCode >= 400 && statusCode < 500) {
        error = CLIENT_ERROR;
      } else if (statusCode >= 500) {
        error = SERVER_ERROR;
      }
    }

    return ofNullable(error);
  }

  public String getErrorMessage() {
    return message;
  }

}
