/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation;

import static com.mulesoft.connectivity.rest.commons.api.error.RestError.CONNECTIVITY;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.config.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.dw.CommonsBindingContext;
import com.mulesoft.connectivity.rest.commons.api.dw.CommonsExpressionLanguage;
import com.mulesoft.connectivity.rest.commons.internal.dw.CommonsExpressionLanguageImpl;
import com.mulesoft.connectivity.rest.commons.internal.util.MediaTypeUtils;

import java.util.concurrent.CompletionException;
import java.util.function.Function;

import javax.inject.Inject;

public abstract class BaseRestOperation {

  @Inject
  private ExpressionLanguage expressionLanguage;

  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  protected void setExpressionLanguage(ExpressionLanguage expressionLanguage) {
    checkArgument(expressionLanguage != null, "ExpressionLanguage cannot be null");
    this.expressionLanguage = expressionLanguage;
  }

  /**
   * Simplified, and limited, version of the runtime's expression language, which heavily relies on {@link CommonsBindingContext}.
   * <br/>
   * Example below:
   *
   * <pre>
   * CommonsBindingContext.Builder builder = CommonsBindingContext.builder();
   * String json = "{\"name\":\"John\", \"age\":30, \"car\":null}";
   * InputStream value = new ByteArrayInputStream(json.getBytes(StandardCharsets.UTF_8));
   * builder.addJson("jsonBinding", value, StandardCharsets.UTF_8);
   * CommonsExpressionLanguageValue evaluatedExpression =
   *     commonsExpressionLanguage.evaluateJson("jsonBinding - 'car'", builder.build());
   * </pre>
   *
   * leaving in {@code evaluatedExpression} a CommonsExpressionLanguageValue object with the following String value:
   *
   * <pre>
   *   {"name":"John", "age":30}
   * </pre>
   *
   * @return a {@code CommonsExpressionLanguage} to evaluate expressions with.
   */
  protected CommonsExpressionLanguage getCommonsExpressionLanguage() {
    return new CommonsExpressionLanguageImpl(expressionLanguage);
  }

  /**
   * Utility method that provides a {@link Function<HttpResponse, HttpResponse>} responsible for handling {@link Throwable}s and
   * notify the {@link CompletionCallback} on non-blocking operations only. It will unwrap a {@link CompletionException} to get
   * the root cause. <br/>
   * Example of usage in a non-blocking operation:
   * 
   * <pre>
   * connection.sendAsync(HttpRequest.builder()
   *     .uri(connection.getBaseUri() + "/accounts/" + accountId)
   *     .build())
   *     .thenAccept(httpResponse -> completionCallback.success(Result.<String, Integer>builder()
   *         .output(IOUtils.toString(httpResponse.getEntity().getContent()))
   *         .attributes(httpResponse.getStatusCode())
   *         .attributesMediaType(APPLICATION_JAVA)
   *         .build()))
   *     .exceptionally(notifyCompletionCallbackError(completionCallback));
   * </pre>
   *
   * @param completionCallback the {@link CompletionCallback} to be notified about the error.
   * @param <P> type for output.
   * @param <A> type for output attributes.
   * @return a {@link Function} to be provided as {@link java.util.concurrent.CompletableFuture#exceptionally(Function)} to handle
   *         any {@link Throwable} that happens during the execution of the completable future. It is responsible for notifying
   *         the {@link CompletionCallback} with the {@link Throwable}.
   */
  protected final <P, A, T> Function<Throwable, T> notifyCompletionCallbackError(CompletionCallback<P, A> completionCallback) {
    return throwable -> {
      Throwable rootCause = throwable;
      if (throwable instanceof CompletionException) {
        rootCause = throwable.getCause();
      }
      if (!(rootCause instanceof ModuleException)) {
        rootCause = toUnknownErrorModuleException(rootCause);
      }
      completionCallback.error(rootCause);
      return null;
    };
  }

  /**
   * Allows to customize the generation of a {@link ModuleException} for unknown errors.
   *
   * @param throwable {@link Throwable} given an unexpected error it should create a {@link ModuleException} with a generic error
   *        like {@link com.mulesoft.connectivity.rest.commons.api.error.RestError#CONNECTIVITY}.
   * @return creates a {@link ModuleException} with a generic error like
   *         {@link com.mulesoft.connectivity.rest.commons.api.error.RestError#CONNECTIVITY}.
   */
  protected ModuleException toUnknownErrorModuleException(Throwable throwable) {
    return new ModuleException(createStaticMessage("Unknown error"), CONNECTIVITY, throwable);
  }

  /**
   * This method specifies the {@link MediaType} that should be assumed the response to have in case the remote service doesn't
   * specify a {@code Content-Type} header. <br/>
   * By default, this method returns {@link MediaType#APPLICATION_JSON} since it's the most common response type. However, this
   * method should be overwritten if a different type of media type is to be expected, or if you know that a certain encoding will
   * be enforced.
   *
   * @return the {@link MediaType} to assign the response in case the server doesn't specify one in its response
   */
  protected MediaType getDefaultResponseMediaType() {
    return APPLICATION_JSON;
  }

  /**
   * Resolves the default {@link MediaType} to be used when processing the response if remote server doesn't specify a
   * {@code Content-Type} header.
   * 
   * @param config {@link RestConfiguration} to get the defaultCharset from Mule Runtime.
   * @return the {@link MediaType}.
   */
  protected final MediaType resolveDefaultResponseMediaType(RestConfiguration config) {
    return MediaTypeUtils.resolveDefaultResponseMediaType(config.getCharset(), getDefaultResponseMediaType());
  }

}
