/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation;

import static java.util.Objects.requireNonNull;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.metadata.MediaType;

import com.mulesoft.connectivity.rest.commons.internal.http.HttpResponseAttributes;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.OptionalLong;

/**
 * Represents a successful result of invoking an {@link org.mule.runtime.http.api.domain.message.request.HttpRequest} to the
 * server.
 *
 * @since 1.0
 */
public class HttpResponseResult {

  private InputStream entityContent;
  private MediaType mediaType = null;
  private HttpResponseAttributes httpResponseAttributes = null;
  private OptionalLong length = OptionalLong.empty();

  public static HttpResponseResult.Builder builder() {
    return new HttpResponseResult.Builder();
  }

  protected HttpResponseResult() {}

  public HttpResponseResult.Builder copy() {
    HttpResponseResult.Builder builder = (new HttpResponseResult.Builder()).entityContent(this.entityContent);
    builder.httpResponseAttributes(this.httpResponseAttributes);
    builder.mediaType(this.mediaType);
    builder.length(this.getByteLength());
    return builder;
  }

  public InputStream getEntityContent() {
    return this.entityContent;
  }

  public HttpResponseAttributes getHttpResponseAttributes() {
    return this.httpResponseAttributes;
  }

  public MediaType getMediaType() {
    return this.mediaType;
  }

  public Charset getCharset() {
    return this.mediaType.getCharset().get();
  }

  public OptionalLong getByteLength() {
    return this.length;
  }

  public static class Builder {

    private static final byte[] NO_BYTES = new byte[0];

    private final HttpResponseResult product = new HttpResponseResult();

    protected Builder() {}

    public HttpResponseResult.Builder entityContent(InputStream entityContent) {
      if (entityContent == null) {
        this.product.entityContent = new ByteArrayInputStream(NO_BYTES);
      } else {
        this.product.entityContent = entityContent;
      }
      return this;
    }

    public HttpResponseResult.Builder httpResponseAttributes(HttpResponseAttributes httpResponseAttributes) {
      this.product.httpResponseAttributes = httpResponseAttributes;
      return this;
    }

    public HttpResponseResult.Builder mediaType(MediaType mediaType) {
      checkArgument(mediaType.getCharset().isPresent(), "'defaultResponseMediaType' should have a charset defined");
      this.product.mediaType = mediaType;
      return this;
    }

    public HttpResponseResult.Builder length(OptionalLong length) {
      this.product.length = length;
      return this;
    }

    public HttpResponseResult build() {
      requireNonNull(this.product.entityContent, "entityContent cannot be null");
      requireNonNull(this.product.mediaType, "mediaType cannot be null");
      requireNonNull(this.product.httpResponseAttributes, "httpResponseAttributes cannot be null");
      return this.product;
    }
  }

}
