/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseResult;
import com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy.PagingStrategy;

import java.io.InputStream;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

/**
 * Provides the base functionality for paginating the results of a REST endpoint.
 * <p>
 * Different APIs use different paging strategies which are tackled in implementations of this class. This class provides the
 * basics as:
 *
 * <ul>
 * <li>Executing one request per page to be obtained</li>
 * <li>Ability to parse the response to extract the data and useful metadata as the total results count (if provided)</li>
 * <li>Handling streaming</li>
 * <li>Handling the nuances of the {@link PagingProvider} contract</li>
 * <li>Handling encodings and media types</li>
 * </ul>
 * <p>
 * Each item in the obtained page will be returned as an individual {@link TypedValue}, with the proper mime type and encoding
 * set.
 * <p>
 * Type parameters: <C> – subclass of {@link RestConnection}
 * 
 * @since 1.0
 */
public abstract class RestPagingProvider<C extends RestConnection> implements PagingProvider<C, TypedValue<String>> {

  private final MediaType defaultResponseMediaType;

  private final Function<C, HttpRequestBuilder> requestFactory;

  private boolean stopPaging = false;

  private final PagingStrategy pagingStrategy;

  /**
   * Creates a new instance
   *
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance.
   * @param defaultResponseMediaType the {@link MediaType} for the page items if the server doesn't specify one.
   * @param pagingStrategy the pagination strategy.
   */
  public RestPagingProvider(Function<C, HttpRequestBuilder> requestFactory,
                            MediaType defaultResponseMediaType,
                            PagingStrategy pagingStrategy) {

    this.requestFactory = requestFactory;
    this.defaultResponseMediaType = defaultResponseMediaType;
    this.pagingStrategy = pagingStrategy;
  }

  @Override
  public final List<TypedValue<String>> getPage(C connection) {
    if (stopPaging) {
      return emptyList();
    }

    HttpRequestBuilder requestBuilder = requestFactory.apply(connection);
    requestBuilder = pagingStrategy.configureRequest(connection.getBaseUri(), requestBuilder, () -> stopPaging());

    if (stopPaging) {
      return emptyList();
    }

    // Send the REQUEST
    HttpResponseResult result = send(connection, requestBuilder, defaultResponseMediaType);

    TypedValue<String> rawPage = toRawPage(result.getEntityContent(), result.getMediaType());

    if (isBlank(rawPage.getValue()) || stopPaging) {
      return emptyList();
    }

    // Post processes response and extract pages
    List<TypedValue<String>> page = extractPageItems(postProcess(rawPage));

    // Calls the onPage hook to do something on every page according to the pagination strategy
    pagingStrategy.onPage(page, rawPage, result.getHttpResponseAttributes().getHeaders(), () -> stopPaging());

    return page;
  }

  /**
   * Given the server response this method will consume the response from the content {@link InputStream} with the given
   * {@link MediaType}, which may or may not contain a charset (cases like application/json should use the default encodings),
   * returns a {@link TypedValue<String>} with the content as {@link String}.
   *
   * @param content the stream response from the server.
   * @param mediaType the {@link MediaType} for this content.
   * @return a {@link TypedValue<String>} with the content as {@link String}.
   */
  protected abstract TypedValue<String> toRawPage(InputStream content, MediaType mediaType);

  /**
   * Does the HTTP call with the given connection for the requestBuilder.
   * 
   * @param connection raw content page as {@link TypedValue<String>}
   * @param requestBuilder the {@link HttpRequestBuilder} to be executed.
   * @param defaultResponseMediaType to be used if the server doesn't specify one with the
   *        {@link org.mule.runtime.http.api.HttpHeaders.Names#CONTENT_TYPE} header.
   * @return a {@link HttpResponse} which represents the server response.
   */
  protected HttpResponseResult send(C connection, HttpRequestBuilder requestBuilder,
                                    MediaType defaultResponseMediaType) {
    try {
      return connection.send(requestBuilder.build(), defaultResponseMediaType);
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage(e.getMessage()), e);
    }
  }

  /**
   * Given the raw content response body this method will extract the payload data as a list of items which represent a page.
   * 
   * @param content raw content page as {@link TypedValue<String>}
   * @return a {@link List<TypedValue<String>>} which represents a page with items.
   */
  protected abstract List<TypedValue<String>> extractPageItems(TypedValue<String> content);

  /**
   * Allows post process the server response before the payload is extracted.
   *
   * @param rawPage the server response as it.
   * @return either the same rawPage or a new page modified.
   */
  protected TypedValue<String> postProcess(TypedValue<String> rawPage) {
    return rawPage;
  }


  @Override
  public Optional<Integer> getTotalResults(RestConnection connection) {
    return empty();
  }

  @Override
  public final void close(RestConnection connection) throws MuleException {}

  /**
   * Context related to the pagination.
   */
  public interface PagingContext {

    /**
     * Allows the strategy to stop the pagination mechanism.
     */
    void stopPaging();

  }

  private void stopPaging() {
    stopPaging = true;
  }

}
