/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy;

import static com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants.HEADERS;
import static com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants.PAYLOAD;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;

/**
 * An implementation for {@link MarkerPagingStrategy} which uses DataWeave to extract the next marker.
 *
 * @since 1.0
 */
public class DataweaveMarkerPagingStrategy extends MarkerPagingStrategy {

  private final String nextMarkerExpression;
  private final ExpressionLanguage expressionLanguage;

  public DataweaveMarkerPagingStrategy(String markerParamName, String nextMarkerExpression,
                                       ExpressionLanguage expressionLanguage) {
    super(markerParamName);
    checkArgument(expressionLanguage != null, "ExpressionLanguage cannot be null");
    checkArgument(isNotBlank(nextMarkerExpression), "nextMarkerExpression cannot be blank");

    this.nextMarkerExpression = nextMarkerExpression;
    this.expressionLanguage = expressionLanguage;
  }

  @Override
  protected String getNextMarker(TypedValue<String> rawPage, MultiMap<String, String> headers) {
    try {
      return (String) expressionLanguage.evaluate(nextMarkerExpression, DataType.STRING,
                                                  BindingContext.builder()
                                                      .addBinding(PAYLOAD, rawPage)
                                                      .addBinding(HEADERS, TypedValue.of(headers))
                                                      .build())
          .getValue();
    } catch (ExpressionRuntimeException e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to extract next marker from expression: %s",
                                                                nextMarkerExpression)),
                                     e);
    }
  }
}
