/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging.strategy;

import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import com.mulesoft.connectivity.rest.commons.api.operation.paging.RestPagingProvider.PagingContext;
import com.mulesoft.connectivity.rest.commons.internal.util.LinkHeaderUtils;

import java.util.List;
import java.util.Map;

/**
 * Base class of {@link PagingStrategy} for APIs which do paging based on a next URL provided in the response of each request.
 * <p>
 * Request number 1 will obtain the first page plus the next URL embedded in the response. Said URL will be the request number 2,
 * which will reply with the next page and a new URL which overwrites the prior one. The process continues until no new URL is
 * provided or the next page comes back empty.
 *
 * @since 1.0
 */
public abstract class HypermediaPagingStrategy implements PagingStrategy {

  private boolean firstPage = true;
  private String nextUrl;

  @Override
  public HttpRequestBuilder configureRequest(String baseUri, HttpRequestBuilder requestBuilder, PagingContext context) {
    if (!firstPage) {
      if (isBlank(nextUrl)) {
        context.stopPaging();
        return requestBuilder;
      }

      HttpRequest baseRequest = requestBuilder.build();

      // Create a new one to allow removing queryParams from the first request
      HttpRequestBuilder builder = HttpRequest.builder()
          .protocol(baseRequest.getProtocol())
          .method(baseRequest.getMethod())
          .entity(baseRequest.getEntity())
          .headers(baseRequest.getHeaders());

      if (isFullUrl()) {
        builder.uri(nextUrl);
      } else {
        builder
            .uri(buildRequestUriWithPath(baseUri, nextUrl))
            .queryParams(baseRequest.getQueryParams());
      }
      return builder;
    }
    return requestBuilder;
  }

  private boolean isFullUrl() {
    return nextUrl.toLowerCase().startsWith("http");
  }

  private String buildRequestUriWithPath(String baseUri, String path) {
    String localPath = path;
    String localBaseUri = baseUri;

    boolean pathStartsWithSlash = localPath.startsWith("/");
    boolean baseEndsInSlash = localBaseUri.endsWith("/");

    if (pathStartsWithSlash && baseEndsInSlash) {
      localPath = localPath.substring(1);
    } else if (!pathStartsWithSlash && !baseEndsInSlash) {
      localBaseUri += '/';
    }

    return localBaseUri + localPath;
  }

  @Override
  public void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage, MultiMap<String, String> headers,
                     PagingContext context) {
    firstPage = false;

    if (isBlank(rawPage.getValue())) {
      nextUrl = null;
      context.stopPaging();
    } else {
      nextUrl = extractNextUrl(rawPage, headers);
    }
  }

  protected abstract String extractNextUrl(TypedValue<String> rawPage, MultiMap<String, String> headers);

  /**
   * Parse Link header into a Map of links by its relation
   *
   * @param linkHeader header to be parsed
   * @return Map of Links by relation
   */
  protected Map<String, String> buildLinkHeaderMap(String linkHeader) {
    return LinkHeaderUtils.buildLinkHeaderMap(linkHeader);
  }

}
