/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.source;

import static com.mulesoft.connectivity.rest.commons.internal.util.SplitPayloadUtils.split;
import static java.lang.String.format;
import static java.util.Objects.isNull;
import static java.util.Optional.of;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.el.ValidationResult;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.runtime.source.PollContext;

import com.mulesoft.connectivity.rest.commons.api.error.SourceStartingException;
import com.mulesoft.connectivity.rest.commons.internal.DWBindingConstants;

import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

public abstract class BaseDataWeaveBasedRestPollingSourceStrategy<S extends Serializable, A> implements
    RestPollingSourceStrategy<S, A> {

  protected final String watermarkExpression;
  protected final String identityExpression;
  protected final String itemsExpression;
  protected final String itemBinding;
  private final ExpressionLanguage expressionLanguage;
  private final Class<S> watermarkDataType;

  public BaseDataWeaveBasedRestPollingSourceStrategy(String itemsExpression, String identityExpression,
                                                     String watermarkExpression,
                                                     String itemBinding, ExpressionLanguage expressionLanguage,
                                                     Class<S> watermarkDataType)
      throws SourceStartingException {
    this.watermarkExpression = watermarkExpression;
    this.identityExpression = identityExpression;
    this.itemsExpression = itemsExpression;
    this.itemBinding = itemBinding;
    this.expressionLanguage = expressionLanguage;
    this.watermarkDataType = watermarkDataType;
    validateExpressionsAndArguments();
  }

  private void validateExpressionsAndArguments() throws SourceStartingException {
    checkArgument(!isBlank(watermarkExpression), "watermarkExpression must not be null or blank");
    checkArgument(!isBlank(itemsExpression), "itemsExpression must not be null or blank");
    checkArgument(!isBlank(identityExpression), "identityExpression must not be null or blank");
    checkArgument(!isBlank(itemBinding), "itemBinding expression must not be null or blank, suggested expression: 'item'");
    checkArgument(!isNull(expressionLanguage), "expressionLanguage must not be null");
    checkArgument(!isNull(watermarkDataType), "watermarkDataType must not be null");
    validateExpression(watermarkExpression);
    validateExpression(identityExpression);
    validateExpression(itemsExpression);
  }

  private void validateExpression(String expression) throws SourceStartingException {
    ValidationResult validationResult = expressionLanguage.validate(expression);

    if (!validationResult.isSuccess()) {
      throw new SourceStartingException(format("Expression is not valid: %s", expression));
    }
  }

  @Override
  public S getItemWatermark(Optional<S> lastWatermark, TypedValue<String> payload,
                            TypedValue<String> item) {
    return (S) expressionLanguage
        .evaluate(watermarkExpression, DataType.fromType(watermarkDataType),
                  getDefaultBindingContextBuilder(payload, lastWatermark, of(item)))
        .getValue();
  }

  @Override
  public String getItemIdentity(Optional<S> lastWatermark, TypedValue<String> payload, TypedValue<String> item) {
    return (String) expressionLanguage
        .evaluate(identityExpression, STRING,
                  getDefaultBindingContextBuilder(payload, lastWatermark, of(item)))
        .getValue();
  }

  @Override
  public List<TypedValue<String>> extractItems(Optional<S> lastWatermark, TypedValue<String> fullResponse,
                                               int statusCode, String reasonPhrase,
                                               MultiMap<String, String> headers) {
    TypedValue<String> items = (TypedValue<String>) expressionLanguage.evaluate(itemsExpression,
                                                                                fullResponse.getDataType(),
                                                                                getDefaultBindingContextBuilder(fullResponse,
                                                                                                                lastWatermark,
                                                                                                                Optional
                                                                                                                    .empty()));
    List<TypedValue<String>> splitList = new ArrayList<>();
    split(expressionLanguage, items).forEachRemaining(item -> {
      splitList.add(new TypedValue<>(item.getValue().toString(), item.getDataType()));
    });
    return splitList;
  }

  @Override
  public Function<PollContext<InputStream, A>, Optional<S>> getLastWatermark() {
    return pollContext -> (Optional<S>) pollContext.getWatermark();
  }

  private BindingContext getDefaultBindingContextBuilder(TypedValue<?> payload, Optional<S> lastWatermark,
                                                         Optional<TypedValue<String>> item) {
    BindingContext.Builder builder = BindingContext.builder()
        .addBinding(DWBindingConstants.PAYLOAD, payload)
        .addBinding(DWBindingConstants.WATERMARK, new TypedValue(lastWatermark, DataType.fromType(watermarkDataType)));

    if (item.isPresent()) {
      builder.addBinding(itemBinding, item.get());
    }
    return builder.build();
  }
}
