/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.source;

import static com.mulesoft.connectivity.rest.commons.internal.util.MediaTypeUtils.resolveDefaultResponseMediaType;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestUtils.closeStream;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestUtils.consumeStringAndClose;
import static java.lang.String.format;
import static java.util.Objects.isNull;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.connectivity.oauth.AccessTokenExpiredException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.source.PollContext;
import org.mule.runtime.extension.api.runtime.source.PollingSource;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;

import com.mulesoft.connectivity.rest.commons.api.config.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseResult;
import com.mulesoft.connectivity.rest.commons.internal.http.HttpResponseAttributes;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Serializable;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base class for implementing a {@link PollingSource<InputStream, A>}.
 *
 * @param <C> {@link RestConnection} type for connection.
 * @param <S> {@link Serializable} type for watermark item.
 * @param <A> {@link Object} type for attributes.
 *
 * @since 1.0
 */
public abstract class RestPollingSource<C extends RestConnection, S extends Serializable, A>
    extends PollingSource<InputStream, A> {

  private static final Logger LOGGER = LoggerFactory.getLogger(RestPollingSource.class);

  @Connection
  protected ConnectionProvider<C> connectionProvider;

  @Inject
  private ExpressionLanguage expressionLanguage;

  protected C connection;

  @Config
  protected RestConfiguration config;

  protected RestPollingSourceStrategy<S, A> restPollingSourceStrategy;

  /**
   * @return The injected expression language
   */
  protected ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  /**
   * This method specifies the {@link RestPollingSourceStrategy} that must be defined as the rest polling source strategy. The
   * rest polling source strategy is needed to process the responses obtained from any SaaS
   *
   * @return the {@link RestPollingSourceStrategy} to be assigned as the strategy of current polling source
   */
  protected abstract RestPollingSourceStrategy<S, A> getRestPollingSourceStrategy() throws MuleException;

  /**
   * This method specifies the {@link MediaType} that should be assumed the response to have in case the remote service doesn't
   * specify a {@code Content-Type} header. <br/>
   * By default, this method returns the configuration media type since it's the most common response type. However, this method
   * should be overwritten if a different type of media type is to be expected, or if you know that a certain encoding will be
   * enforced.
   *
   * @return the {@link MediaType} to assign the response in case the server doesn't specify one in its response
   */
  protected MediaType getDefaultResponseMediaType() {
    return APPLICATION_JSON;
  }

  @Override
  protected void doStart() throws MuleException {
    connection = connectionProvider.connect();
    restPollingSourceStrategy = getRestPollingSourceStrategy();
  }

  @Override
  protected void doStop() {
    connectionProvider.disconnect(connection);
  }

  private HttpResponseResult send(HttpRequest httpRequest, PollContext<InputStream, A> pollContext, MediaType defaultMediaType) {
    String triggerIdentifier = getClass().getSimpleName();
    try {
      return connection.send(httpRequest, defaultMediaType);
    } catch (AccessTokenExpiredException e) {
      pollContext.onConnectionException(new ConnectionException(e));
      throw new MuleRuntimeException(createStaticMessage(format("Exception found while executing poll: '%s'. Access token expiration. "
          + "'%s'", triggerIdentifier, e.getMessage())), e);
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage(format("Exception found while executing poll: '%s'. "
          + "'%s'", triggerIdentifier, e.getMessage())), e);
    }
  }

  @Override
  public void poll(PollContext<InputStream, A> pollContext) {
    final Optional<S> lastWatermark = restPollingSourceStrategy.getLastWatermark().apply(pollContext);
    HttpResponseResult result = send(getRequest(lastWatermark),
                                     pollContext,
                                     resolveDefaultResponseMediaType(config, getDefaultResponseMediaType()));

    TypedValue<String> rawPage = consumeStringAndClose(result.getEntityContent(),
                                                       result.getMediaType(),
                                                       result.getCharset());
    if (!isBlank(rawPage.getValue())) {
      HttpResponseAttributes httpResponseAttributes = result.getHttpResponseAttributes();
      List<TypedValue<String>> items = restPollingSourceStrategy.extractItems(lastWatermark, rawPage,
                                                                              httpResponseAttributes.getStatusCode(),
                                                                              httpResponseAttributes.getReasonPhrase(),
                                                                              httpResponseAttributes.getHeaders());
      if (isNull(items)) {
        throw new IllegalArgumentException("Extracted items must not be null. An empty list must be returned instead of null.");
      }
      for (TypedValue<String> item : items) {
        pollContext.accept(getPollItemConsumer(lastWatermark, rawPage, item,
                                               (A) restPollingSourceStrategy
                                                   .getItemAttributes(httpResponseAttributes.getStatusCode(),
                                                                      httpResponseAttributes.getReasonPhrase(),
                                                                      httpResponseAttributes.getHeaders(),
                                                                      item)));
      }
    }
  }

  private Consumer<PollContext.PollItem<InputStream, A>> getPollItemConsumer(Optional<S> lastWatermark,
                                                                             TypedValue<String> fullResponse,
                                                                             TypedValue<String> item,
                                                                             A itemAttributes) {
    return pollItem -> {
      Result<InputStream, A> itemResult =
          Result.<InputStream, A>builder()
              .output(new ByteArrayInputStream(item.getValue().getBytes()))
              .mediaType(item.getDataType().getMediaType())
              .attributes(itemAttributes)
              .build();

      pollItem.setResult(itemResult);
      pollItem.setWatermark(restPollingSourceStrategy.getItemWatermark(lastWatermark, fullResponse, item));
      pollItem.setId(restPollingSourceStrategy.getItemIdentity(lastWatermark, fullResponse, item));
    };
  }

  @Override
  public void onRejectedItem(Result<InputStream, A> result,
                             SourceCallbackContext callbackContext) {
    if (result.getOutput() != null) {
      closeStream(result.getOutput());
    }
    LOGGER.debug("Item Rejected");
  }

  /**
   * This method generates the http request to be executed for the polling source at each execution time
   * 
   * @param lastWatermark latest watermark obtained
   * @return the {@link HttpRequest} to be executed
   */
  protected abstract HttpRequest getRequest(Optional<S> lastWatermark);


}
