/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.source;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.runtime.source.PollContext;

import java.io.InputStream;
import java.io.Serializable;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

/**
 * This interface defines the necessary methods to extract the information needed to use a {@link RestPollingSource}
 */
public interface RestPollingSourceStrategy<S extends Serializable, A> {

  /**
   * This method gets the item watermark from the SaaS response having the full response, the individual item and the last
   * watermark available.
   * 
   * @param lastWatermark latest watermark obtained
   * @param payload current payload obtained by SaaS response
   * @param item current item
   *
   * @return the {@link S} watermark defined as generic type
   */
  S getItemWatermark(Optional<S> lastWatermark, TypedValue<String> payload,
                     TypedValue<String> item);

  /**
   * This method gets the item identity from the SaaS response having the full response, the individual item and the last
   * watermark available.
   * 
   * @param lastWatermark latest watermark obtained
   * @param payload current payload obtained by SaaS response
   * @param item current item
   *
   * @return the {@link String} identity as string type
   */
  String getItemIdentity(Optional<S> lastWatermark, TypedValue<String> payload,
                         TypedValue<String> item);

  /**
   * This method gets the list of items from the SaaS response.
   * 
   * @param lastWatermark latest watermark obtained
   * @param fullResponse current payload obtained by SaaS response
   * @param statusCode http status code from SaaS response
   * @param reasonPhrase http reason phrase from SaaS response
   * @param headers http headers from SaaS response
   *
   * @return the {@link List<TypedValue<String>>} list of items
   */
  List<TypedValue<String>> extractItems(Optional<S> lastWatermark, TypedValue<String> fullResponse,
                                        int statusCode, String reasonPhrase,
                                        MultiMap<String, String> headers);

  /**
   * This method gets the latest watermark from the poll context to be used as starting point for new responses.
   *
   * @return the {@link Function< PollContext <InputStream, A>, Optional<S>>} identity as string type
   */
  Function<PollContext<InputStream, A>, Optional<S>> getLastWatermark();

  /**
   * This method returns the attributes to be set for the item result attributes.
   * 
   * @param item current item obtained by SaaS response
   * @param statusCode http status code from SaaS response
   * @param reasonPhrase http reason phrase from SaaS response
   * @param headers http headers from SaaS response
   *
   * @return the {@link A} identity as string type
   */
  A getItemAttributes(int statusCode, String reasonPhrase,
                      MultiMap<String, String> headers, TypedValue<String> item);
}
