/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.dw;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_XML;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;

import com.mulesoft.connectivity.rest.commons.api.dw.CommonsBindingContext;
import com.mulesoft.connectivity.rest.commons.api.dw.CommonsExpressionLanguageValue;

import java.io.InputStream;
import java.nio.charset.Charset;

public class CommonsBindingContextBuilder implements CommonsBindingContext.Builder {

  private final BindingContext.Builder builder;

  public CommonsBindingContextBuilder() {
    this.builder = BindingContext.builder();
  }

  @Override
  public CommonsBindingContextBuilder addContent(String identifier, InputStream value, MediaType mediaType) {
    checkArgument(mediaType.getCharset().isPresent(), "'mediaType' should have a charset defined");
    return addBinding(identifier, value,
                      DataType.builder().type(InputStream.class).mediaType(mediaType).build());
  }

  @Override
  public CommonsBindingContextBuilder addJson(String identifier, InputStream value, Charset charset) {
    return addBinding(identifier, value,
                      DataType.builder().type(InputStream.class).mediaType(APPLICATION_JSON.withCharset(charset)).build());
  }

  @Override
  public CommonsBindingContextBuilder addJson(String identifier, CommonsExpressionLanguageValue value) {
    return addBinding(identifier, value.asString(), DataType.JSON_STRING);
  }

  @Override
  public CommonsBindingContextBuilder addXml(String identifier, InputStream value, Charset charset) {
    return addBinding(identifier, value,
                      DataType.builder().type(InputStream.class).mediaType(APPLICATION_XML.withCharset(charset)).build());
  }

  @Override
  public CommonsBindingContextBuilder addXml(String identifier, CommonsExpressionLanguageValue value) {
    return addBinding(identifier, value.asString(), DataType.XML_STRING);
  }

  @Override
  public CommonsBindingContextBuilder addJava(String identifier, Object value) {
    DataType object = DataType.fromObject(value);
    return addBinding(identifier, value, object);
  }

  @Override
  public CommonsBindingContext build() {
    return new CommonsBindingContextImpl(builder.build());
  }

  private CommonsBindingContextBuilder addBinding(String identifier, Object value, DataType dataType) {
    checkArgument(isNotBlank(identifier), "Binding identifier cannot be blank");
    checkArgument(value != null, "Binding value cannot be null");
    checkArgument(dataType != null, "Binding dataType cannot be null");

    TypedValue typedValue = new TypedValue(value, dataType);
    builder.addBinding(identifier, typedValue);
    return this;
  }

  private class CommonsBindingContextImpl implements CommonsBindingContext {

    private final BindingContext bindingContext;

    private CommonsBindingContextImpl(BindingContext bindingContext) {
      this.bindingContext = bindingContext;
    }

    @Override
    public BindingContext bindingContext() {
      return bindingContext;
    }
  }

}
