/*
 * (c) 2003-2022 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.http;

import static java.lang.System.lineSeparator;
import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;

import java.util.Map;
import java.util.stream.Stream;

/**
 * Internal representation of an HTTP response message attributes. This class should not be used as part of the connector's
 * implementation API.
 *
 * @since 1.0
 */
public class HttpResponseAttributes {

  private static final String TAB = "   ";
  private static final String DOUBLE_TAB = TAB + TAB;

  /**
   * HTTP status code of the response.
   */
  private final int statusCode;

  /**
   * Map of HTTP headers in the message.
   */
  private MultiMap<String, String> headers;

  /**
   * HTTP reason phrase of the response.
   */
  private final String reasonPhrase;

  public HttpResponseAttributes(int statusCode, String reasonPhrase, MultiMap<String, String> headers) {
    this.headers = headers != null ? new CaseInsensitiveMultiMap(headers) : emptyMultiMap();
    this.statusCode = statusCode;
    this.reasonPhrase = reasonPhrase;
  }

  public int getStatusCode() {
    return statusCode;
  }

  public String getReasonPhrase() {
    return reasonPhrase;
  }

  public MultiMap<String, String> getHeaders() {
    return headers;
  }

  public String toString() {
    StringBuilder builder = new StringBuilder();

    builder.append(this.getClass().getName()).append(lineSeparator()).append("{").append(lineSeparator())
        .append(TAB).append("Status Code=").append(statusCode).append(lineSeparator())
        .append(TAB).append("Reason Phrase=").append(reasonPhrase).append(lineSeparator());

    buildMapToString(headers, "Headers", headers.entryList().stream(), builder);

    builder.append("}");

    return builder.toString();
  }

  private String formatHttpAttributesMapsToString(String name, Stream<Map.Entry<String, String>> stream) {
    StringBuilder builder = new StringBuilder();
    builder.append(TAB).append(name).append("=[").append(lineSeparator());
    stream.forEach(element -> builder.append(DOUBLE_TAB)
        .append(element.getKey()).append("=").append(obfuscateValueIfNecessary(element)).append(lineSeparator()));
    builder.append(TAB).append("]").append(lineSeparator());
    return builder.toString();
  }

  private String obfuscateValueIfNecessary(Map.Entry<String, String> entry) {
    String key = entry.getKey();
    if (key.equals("password") || key.equals("pass") || key.contains("secret") || key.contains("authorization")) {
      return "****";
    } else {
      return entry.getValue();
    }
  }

  private StringBuilder buildMapToString(Map<?, ?> map, String name, Stream<Map.Entry<String, String>> stream,
                                         StringBuilder builder) {
    if (map.isEmpty()) {
      builder.append(TAB).append(name).append("=[]").append(lineSeparator());
      return builder;
    }
    builder.append(formatHttpAttributesMapsToString(name, stream));
    return builder;
  }
}
