/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.source;

import static com.mulesoft.connectivity.rest.commons.internal.util.RequestStreamingUtils.doRequestAndConsumeString;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.stream.StreamSupport.stream;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.closeStream;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.consumeStringAndClose;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.resolveCharset;
import static org.mule.runtime.api.el.BindingContext.builder;
import static org.mule.runtime.api.metadata.DataType.JSON_STRING;
import static org.mule.runtime.api.metadata.DataType.STRING;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.error.SourceStartingException;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.el.ValidationResult;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.core.internal.streaming.bytes.ByteArrayCursorStreamProvider;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.source.PollContext;
import org.mule.runtime.extension.api.runtime.source.PollingSource;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

import com.mulesoft.connectivity.rest.commons.api.configuration.StreamingType;
import com.mulesoft.connectivity.rest.commons.internal.util.DwUtils;
import com.mulesoft.connectivity.rest.commons.internal.util.FromCursorProviderInputStream;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base class for defining a polling source that consumes a remote REST endpoint in a connector.
 *
 * @since 1.0
 */
public abstract class RestPollingSource
    extends PollingSource<TypedValue<InputStream>, HttpResponseAttributes> {

  private static final Logger LOGGER = LoggerFactory.getLogger(RestPollingSource.class);

  @Config
  private RestConfiguration config;

  @Connection
  private ConnectionProvider<RestConnection> connectionProvider;

  @Inject
  private ExpressionLanguage expressionLanguage;

  protected RestConnection connection;

  protected final String watermarkExpression;
  protected final String identityExpression;
  protected final String itemsExpression;
  protected final String requestBodyExpression;

  public RestPollingSource(String itemsExpression, String watermarkExpression, String identityExpression,
                           String requestBodyExpression) {
    requireNonNull(watermarkExpression);
    requireNonNull(itemsExpression);

    this.watermarkExpression = watermarkExpression;
    this.identityExpression = identityExpression;
    this.itemsExpression = itemsExpression;
    this.requestBodyExpression = requestBodyExpression;
  }

  @Override
  protected void doStart() throws MuleException {
    validateExpression(watermarkExpression);
    validateExpression(identityExpression);
    validateExpression(itemsExpression);
    validateExpression(requestBodyExpression);

    for (String bindingExpression : getParameterBinding().getAllBindingExpressions()) {
      if (DwUtils.isExpression(bindingExpression)) {
        validateExpression(bindingExpression);
      }
    }

    connection = connectionProvider.connect();
  }

  @Override
  protected void doStop() {
    connectionProvider.disconnect(connection);
  }

  /**
   * Returns the parameter binding configuration of this RestPollingSource.
   */
  protected abstract RequestParameterBinding getParameterBinding();

  /**
   * Returns a MultiMap containing all the parameters this source exposes to the user.
   * Each entry must contain the parameter name as key and its value represented as a TypedValue.
   */
  protected abstract MultiMap<String, TypedValue<?>> getParameterValues();

  /**
   * Returns the request path for this RestPollingSource with placeholders for its uri parameters.
   * i.e: /user/{username}/events
   */
  protected abstract String getPathTemplate();

  /**
   * Return a RequestBuilder configured to do the request to the endpoint this source must poll.
   * BaseUri, Path and Method must be configured.
   * @param path The request path with the placeholders replaced with its corresponding values.
   */
  protected abstract RestRequestBuilder getRequestBuilder(String path);

  @Override
  public void poll(PollContext<TypedValue<InputStream>, HttpResponseAttributes> pollContext) {
    final Serializable watermark = pollContext.getWatermark().orElse(null);

    final Result<TypedValue<String>, HttpResponseAttributes> result =
        doRequestAndConsumeString(connection, config, getRestRequestBuilder(watermark), getDefaultResponseMediaType());

    for (TypedValue<CursorStreamProvider> item : getItems(result.getOutput(), watermark)) {
      pollContext.accept(getPollItemConsumer(watermark, result, item));
    }
  }

  private RestRequestBuilder getRestRequestBuilder(Serializable watermark) {
    RequestParameterBinding parameterBinding = getParameterBinding();
    RestRequestBuilder requestBuilder =
        getRequestBuilder(buildRequestPath(getPathTemplate(), parameterBinding.getUriParams(), watermark));

    TypedValue<InputStream> requestBody = getRequestBody(watermark);
    if (requestBody != null) {
      requestBuilder.setBody(requestBody, StreamingType.AUTO);
    }

    parameterBinding
        .getHeaders()
        .forEach(i -> requestBuilder.addHeader(i.getKey(), getParameterValue(i.getValue(), watermark)));
    parameterBinding
        .getQueryParams()
        .forEach(i -> requestBuilder.addQueryParam(i.getKey(), getParameterValue(i.getValue(), watermark)));
    return requestBuilder;
  }

  private Consumer<PollContext.PollItem<TypedValue<InputStream>, HttpResponseAttributes>> getPollItemConsumer(Serializable watermark,
                                                                                                              Result<TypedValue<String>, HttpResponseAttributes> result,
                                                                                                              TypedValue<CursorStreamProvider> item) {
    return pollItem -> {
      TypedValue<InputStream> inputStreamTypedValue =
          new TypedValue<>(FromCursorProviderInputStream.of(item.getValue()), item.getDataType());

      Result<TypedValue<InputStream>, HttpResponseAttributes> itemResult =
          Result.<TypedValue<InputStream>, HttpResponseAttributes>builder()
              .output(inputStreamTypedValue)
              .attributes(result.getAttributes().orElse(null))
              .build();

      pollItem.setResult(itemResult);

      if (isNotBlank(watermarkExpression)) {
        pollItem.setWatermark(getItemWatermark(result.getOutput(), watermark, item));
      }

      if (isNotBlank(identityExpression)) {
        pollItem.setId(getIdentity(result.getOutput(), watermark, item));
      }
    };
  }

  private String buildRequestPath(String pathTemplate, List<RequestParameterBinding.Binding> uriParams, Serializable watermark) {
    if (uriParams.isEmpty()) {
      return pathTemplate;
    }

    Map<String, String> uriParamValues = new HashMap<>();

    uriParams.forEach(i -> uriParamValues.put(i.getKey(), getParameterValue(i.getValue(), watermark)));

    String path = pathTemplate;
    for (String key : uriParamValues.keySet()) {
      path = path.replace("{" + key + "}", uriParamValues.get(key));
    }

    return path;
  }

  private String getParameterValue(String expression, Serializable watermark) {
    if (!DwUtils.isExpression(expression)) {
      return expression;
    }

    return (String) expressionLanguage
        .evaluate(expression, STRING, buildContext(null, watermark, null))
        .getValue();
  }

  protected DataType getWatermarkDataType() {
    return STRING;
  }

  private Serializable getItemWatermark(TypedValue<String> payload, Serializable currentWatermark,
                                        TypedValue<CursorStreamProvider> item) {
    return (Serializable) expressionLanguage
        .evaluate(watermarkExpression, getWatermarkDataType(), buildContext(payload, currentWatermark, item))
        .getValue();
  }

  private String getIdentity(TypedValue<String> payload, Serializable currentWatermark, TypedValue<CursorStreamProvider> item) {
    return (String) expressionLanguage
        .evaluate(identityExpression, STRING, buildContext(payload, currentWatermark, item))
        .getValue();
  }

  protected DataType getRequestBodyDataType() {
    return JSON_STRING;
  }

  private TypedValue<InputStream> getRequestBody(Serializable currentWatermark) {
    if (isNotBlank(requestBodyExpression)) {
      TypedValue<?> body =
          expressionLanguage.evaluate(requestBodyExpression, getRequestBodyDataType(),
                                      buildContext(null, currentWatermark, null));

      TypedValue<String> stringTypedValue = consumeStringAndClose(body.getValue(), getDefaultResponseMediaType(),
                                                                  resolveCharset(empty(), getDefaultResponseMediaType()));

      return new TypedValue<>(new ByteArrayInputStream(stringTypedValue.getValue().getBytes()), body.getDataType());
    }

    return null;
  }

  private List<TypedValue<CursorStreamProvider>> getItems(TypedValue<String> payload, Serializable currentWatermark) {
    TypedValue<?> result =
        expressionLanguage.evaluate(itemsExpression, buildContext(payload, currentWatermark, null));

    Iterator<TypedValue<?>> splitResult =
        expressionLanguage.split("#[payload default []]", buildSplitContext(result));

    Iterable<TypedValue<?>> iterable = () -> splitResult;
    return stream(iterable.spliterator(), false)
        .map(this::getCursorStreamProviderValueFromSplitResult)
        .collect(Collectors.toList());
  }

  private TypedValue<CursorStreamProvider> getCursorStreamProviderValueFromSplitResult(TypedValue<?> typedValue) {
    if (typedValue.getValue() instanceof CursorStreamProvider) {
      return new TypedValue<>((CursorStreamProvider) typedValue.getValue(), typedValue.getDataType());
    } else if (typedValue.getValue() instanceof String) {
      return new TypedValue<>(new ByteArrayCursorStreamProvider(((String) typedValue.getValue()).getBytes()),
                              typedValue.getDataType());
    }
    throw new IllegalArgumentException("Could not create CursorStreamProvider for type: " + typedValue.getValue().getClass());
  }

  private void validateExpression(String expression) throws SourceStartingException {
    if (isBlank(expression)) {
      return;
    }

    ValidationResult validationResult = expressionLanguage.validate(expression);

    if (!validationResult.isSuccess()) {
      throw new SourceStartingException(format("Expression is not valid: %s", expression));
    }
  }

  private BindingContext buildSplitContext(TypedValue<?> payload) {
    return builder().addBinding("payload", TypedValue.of(payload)).build();
  }

  private BindingContext buildContext(
                                      TypedValue<?> payload, Serializable currentWatermark,
                                      TypedValue<CursorStreamProvider> item) {
    BindingContext.Builder builder =
        builder()
            .addBinding("payload", payload)
            .addBinding("watermark", TypedValue.of(currentWatermark))
            .addBinding("parameters", TypedValue.of(getParameterValues()));

    if (item != null) {
      builder.addBinding("item", item);
    }

    return builder.build();
  }

  @Override
  public void onRejectedItem(Result<TypedValue<InputStream>, HttpResponseAttributes> result,
                             SourceCallbackContext callbackContext) {
    if (result.getOutput() != null) {
      closeStream(result.getOutput().getValue());
    }
    LOGGER.debug("Item Rejected");
  }

  protected MediaType getDefaultResponseMediaType() {
    return MediaType.APPLICATION_JSON;
  }
}
