/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.data.sample;

import static com.mulesoft.connectivity.rest.commons.internal.util.RequestStreamingUtils.doRequestAndConsumeString;
import static org.mule.runtime.api.el.BindingContext.builder;
import static org.mule.runtime.api.metadata.DataType.STRING;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.sdk.api.data.sample.SampleDataException;
import org.mule.sdk.api.data.sample.SampleDataProvider;
import org.mule.sdk.api.runtime.operation.Result;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.api.source.RequestParameterBinding;
import com.mulesoft.connectivity.rest.commons.internal.adapter.SdkResultAdapter;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class RestSampleDataProvider<T, A> implements SampleDataProvider<T, A> {

  private static final Logger LOGGER = LoggerFactory.getLogger(RestSampleDataProvider.class);

  @Config
  private RestConfiguration config;

  @Connection
  protected RestConnection connection;

  @Inject
  protected ExpressionLanguage expressionLanguage;

  @Override
  public String getId() {
    return getClass().getSimpleName();
  }

  @Override
  public Result<T, A> getSample() throws SampleDataException {
    final org.mule.runtime.extension.api.runtime.operation.Result<TypedValue<String>, HttpResponseAttributes> resultCall =
        getResult();

    Result<T, A> result = SdkResultAdapter.from(resultCall);
    return result;
  }

  protected org.mule.runtime.extension.api.runtime.operation.Result<TypedValue<String>, HttpResponseAttributes> getResult() {
    return doRequestAndConsumeString(connection, config, getRestRequestBuilder(), getDefaultResponseMediaType());
  }

  /**
   * Returns the request path for this RestSampleDataProvider with placeholders for its uri parameters.
   * i.e: /user/{username}/events
   */
  protected abstract String getPathTemplate();

  /**
   * Return a RequestBuilder configured to do the request to the endpoint this source must poll.
   * BaseUri, Path and Method must be configured.
   *
   * @param path The request path with the placeholders replaced with its corresponding values.
   */
  protected abstract RestRequestBuilder getRequestBuilder(String path);

  /**
   * Returns the parameter binding configuration of this RestPollingSource.
   */
  protected abstract RequestParameterBinding getParameterBinding();

  /**
   * Returns a MultiMap containing all the parameters this source exposes to the user.
   * Each entry must contain the parameter name as key and its value represented as a TypedValue.
   */
  protected abstract MultiMap<String, TypedValue<?>> getParameterValues();


  protected MediaType getDefaultResponseMediaType() {
    return MediaType.APPLICATION_JSON;
  }

  private RestRequestBuilder getRestRequestBuilder() {
    RequestParameterBinding parameterBinding = getParameterBinding();
    RestRequestBuilder requestBuilder =
        getRequestBuilder(buildRequestPath(getPathTemplate(), parameterBinding.getUriParams()));

    parameterBinding
        .getHeaders()
        .forEach(i -> requestBuilder.addHeader(i.getKey(), getParameterValue(i.getValue())));
    parameterBinding
        .getQueryParams()
        .forEach(i -> requestBuilder.addQueryParam(i.getKey(), getParameterValue(i.getValue())));
    return requestBuilder;
  }

  private String buildRequestPath(String pathTemplate, List<RequestParameterBinding.Binding> uriParams) {
    if (uriParams.isEmpty()) {
      return pathTemplate;
    }

    Map<String, String> uriParamValues = new HashMap<>();

    uriParams.forEach(i -> uriParamValues.put(i.getKey(), getParameterValue(i.getValue())));

    String path = pathTemplate;
    for (String key : uriParamValues.keySet()) {
      path = path.replace("{" + key + "}", uriParamValues.get(key));
    }

    return path;
  }

  private String getParameterValue(String expression) {
    return (String) expressionLanguage
        .evaluate(expression, STRING, buildContext(null))
        .getValue();
  }

  protected BindingContext buildContext(TypedValue<?> payload) {

    BindingContext.Builder builder =
        builder()
            .addBinding("payload", payload);
    return builder.build();
  }
}
