/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import com.mulesoft.connectivity.rest.commons.api.connection.validation.ConnectionValidationSettings;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.api.error.RequestException;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.client.HttpClient;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;

/**
 * Generic contract for connections that allow consuming a Rest API by the use of an underlying {@link HttpClient}
 *
 * @since 1.0
 */
public interface RestConnection {

  /**
   * Sends a request and returns a {@link CompletableFuture} with a {@link Result} which output is the response body in
   * the form of an {@link InputStream} and a {@link HttpResponseAttributes}.
   * <p>
   * In case of failure, the future will be exceptionally completed with a {@link RequestException}, which
   * {@link RequestException#getErrorMessage()} will have its payload made repeatable through the use of the
   * {@code streamingHelper}
   *
   * @param requestBuilder        the {@link RestRequestBuilder} on which the request has been configured
   * @param responseTimeoutMillis the request timeout in milliseconds
   * @param streamingHelper       the {@link StreamingHelper} in case an error message needs to have its payload made repeatable
   * @return a {@link CompletableFuture}
   */
  CompletableFuture<Result<InputStream, HttpResponseAttributes>> request(RestRequestBuilder requestBuilder,
                                                                         int responseTimeoutMillis,
                                                                         MediaType defaultResponseMediaType,
                                                                         StreamingHelper streamingHelper);


  /**
   * Sends a request for which a response entity is not expected. It returns a {@link CompletableFuture} with
   * a {@link Result} which holds an empty String as output and a {@link HttpResponseAttributes}.
   * <p>
   * In case of failure, the future will be exceptionally completed with a {@link RequestException}, which
   * {@link RequestException#getErrorMessage()} will have its payload made repeatable through the use of the
   * {@code streamingHelper}
   *
   * @param requestBuilder        the {@link RestRequestBuilder} on which the request has been configured
   * @param responseTimeoutMillis the request timeout in milliseconds
   * @param streamingHelper       the {@link StreamingHelper} in case an error message needs to have its payload made repeatable
   * @return a {@link CompletableFuture}
   */
  CompletableFuture<Result<String, HttpResponseAttributes>> bodylessRequest(RestRequestBuilder requestBuilder,
                                                                            int responseTimeoutMillis,
                                                                            MediaType defaultResponseMediaType,
                                                                            StreamingHelper streamingHelper);


  /**
   * Sends a request as configured using provided settings builder and validates it using the provided conditions.
   * Returns a {@link ConnectionValidationResult} corresponding to this validation result.
   * @param settings              the settings that indicate how to do this request and the validations to do in it's response.
   * @param responseTimeoutMillis the request timeout in milliseconds
   * @return A {@link ConnectionValidationResult} corresponding to this validation result
   */
  ConnectionValidationResult validate(ConnectionValidationSettings settings, int responseTimeoutMillis);

  /**
   * @return the base uri of the REST API being consumed
   */
  String getBaseUri();

  /**
   * Stops this connection by freeing resources and stopping the underlying {@link HttpClient}
   */
  void stop();

}
