/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.datasense.valueprovider;

import static com.mulesoft.connectivity.rest.commons.internal.util.RequestStreamingUtils.doRequestAndConsumeString;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static java.util.Objects.requireNonNull;
import static org.mule.runtime.api.el.BindingContext.builder;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.extension.api.values.ValueBuilder.newValue;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.api.source.RequestParameterBinding;
import com.mulesoft.connectivity.rest.commons.internal.util.RequestStreamingUtils;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.inject.Inject;

/**
 * An implementation of a {@link ValueProvider} that consumes a remote endpoint and extracts the values and its names using the
 * provided expressions.
 *
 * @since 1.0
 */
public abstract class RestValueProvider implements ValueProvider {

  @Config
  private RestConfiguration config;

  @Connection
  protected RestConnection connection;

  @Inject
  private ExpressionLanguage expressionLanguage;

  protected final String itemsExpression;
  protected final String itemValueExpression;
  protected final String itemNameExpression;

  public RestValueProvider(String itemsExpression,
                           String itemValueExpression,
                           String itemNameExpression) {

    requireNonNull(itemsExpression);
    requireNonNull(itemValueExpression);

    this.itemsExpression = itemsExpression;
    this.itemValueExpression = itemValueExpression;
    this.itemNameExpression = itemNameExpression;
  }

  /**
   * Returns the parameter binding configuration of this RestPollingSource.
   */
  protected abstract RequestParameterBinding getParameterBinding();

  /**
   * Returns the request path for this RestPollingSource with placeholders for its uri parameters.
   * i.e: /user/{username}/events
   */
  protected abstract String getPathTemplate();

  /**
   * Return a RequestBuilder configured to do the request to the endpoint this source must poll.
   * BaseUri, Path and Method must be configured.
   * @param path The request path with the placeholders replaced with its corresponding values.
   */
  protected abstract RestRequestBuilder getRequestBuilder(String path);

  private RestRequestBuilder getRestRequestBuilder() {
    RequestParameterBinding parameterBinding = getParameterBinding();
    RestRequestBuilder requestBuilder =
        getRequestBuilder(buildRequestPath(getPathTemplate(), parameterBinding.getUriParams()));

    parameterBinding
        .getHeaders()
        .forEach(i -> requestBuilder.addHeader(i.getKey(), getParameterValue(i.getValue())));
    parameterBinding
        .getQueryParams()
        .forEach(i -> requestBuilder.addQueryParam(i.getKey(), getParameterValue(i.getValue())));
    return requestBuilder;
  }

  private String buildRequestPath(String pathTemplate, List<RequestParameterBinding.Binding> uriParams) {
    if (uriParams.isEmpty()) {
      return pathTemplate;
    }

    Map<String, String> uriParamValues = new HashMap<>();

    uriParams.forEach(i -> uriParamValues.put(i.getKey(), getParameterValue(i.getValue())));

    String path = pathTemplate;
    for (String key : uriParamValues.keySet()) {
      path = path.replace("{" + key + "}", uriParamValues.get(key));
    }

    return path;
  }

  private String getParameterValue(String expression) {
    return (String) expressionLanguage
        .evaluate(expression, STRING, buildContext(null, null))
        .getValue();
  }

  private BindingContext buildContext(TypedValue<?> payload,
                                      TypedValue<CursorStreamProvider> item) {

    BindingContext.Builder builder =
        builder()
            .addBinding("payload", payload)
            .addBinding("parameter", TypedValue.of(getParameterValues()));

    if (item != null) {
      builder.addBinding("item", item);
    }

    return builder.build();
  }

  /**
   * Returns a MultiMap containing all the parameters this source exposes to the user.
   * Each entry must contain the parameter name as key and its value represented as a TypedValue.
   */
  protected abstract MultiMap<String, TypedValue<?>> getParameterValues();

  @Override
  public Set<Value> resolve() throws ValueResolvingException {
    final Result<TypedValue<String>, HttpResponseAttributes> result =
        doRequestAndConsumeString(connection, config, getRestRequestBuilder(), getDefaultResponseMediaType());

    final Set<Value> values = new HashSet<>();

    for (TypedValue<CursorStreamProvider> item : getItems(result.getOutput())) {
      String itemValue = evaluateExpressionOnItem(item, itemValueExpression);

      String itemName = itemValue;
      if (isNotBlank(itemNameExpression)) {
        itemName = evaluateExpressionOnItem(item, itemNameExpression);
      }

      values.add(newValue(itemValue).withDisplayName(itemName).build());
    }

    return values;
  }

  private String evaluateExpressionOnItem(TypedValue<CursorStreamProvider> item, String expression) {
    return (String) expressionLanguage
        .evaluate(expression, STRING, buildContext(null, item))
        .getValue();
  }

  protected MediaType getDefaultResponseMediaType() {
    return MediaType.APPLICATION_JSON;
  }

  private List<TypedValue<CursorStreamProvider>> getItems(TypedValue<String> payload) {
    TypedValue<?> result =
        expressionLanguage.evaluate(itemsExpression, buildContext(payload, null));

    Iterator<TypedValue<?>> splitResult =
        expressionLanguage.split("#[payload default []]", buildContext(result, null));

    Iterable<TypedValue<?>> iterable = () -> splitResult;
    return StreamSupport.stream(iterable.spliterator(), false)
        .map(RequestStreamingUtils::getCursorStreamProviderValueFromSplitResult)
        .collect(Collectors.toList());
  }
}
