/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;

import java.io.IOException;
import java.io.InputStream;

import org.slf4j.Logger;

/**
 * An implementation of InputStream that wraps a CursorProvider.
 * Manages opening, closing and releasing the resources of the wrapped CursorProvider.
 *
 * @since 1.0
 */
public class FromCursorProviderInputStream extends InputStream {

  private static final Logger LOGGER = getLogger(FromCursorProviderInputStream.class);

  private final CursorStreamProvider cursorProvider;
  private final InputStream stream;

  public static FromCursorProviderInputStream of(CursorStreamProvider cursorProvider) {
    CursorStream stream = cursorProvider.openCursor();
    return new FromCursorProviderInputStream(cursorProvider, stream);
  }

  private FromCursorProviderInputStream(CursorStreamProvider cursorProvider, InputStream stream) {
    this.cursorProvider = cursorProvider;
    this.stream = stream;
  }

  @Override
  public int read(byte[] b) throws IOException {
    return stream.read(b);
  }

  @Override
  public int read(byte[] b, int off, int len) throws IOException {
    return stream.read(b, off, len);
  }

  @Override
  public long skip(long n) throws IOException {
    return stream.skip(n);
  }

  @Override
  public int available() throws IOException {
    return stream.available();
  }

  @Override
  public void mark(int readlimit) {
    stream.mark(readlimit);
  }

  @Override
  public void reset() throws IOException {
    stream.reset();
  }

  @Override
  public boolean markSupported() {
    return stream.markSupported();
  }

  @Override
  public int read() throws IOException {
    return stream.read();
  }

  public void close() throws IOException {
    try {
      stream.close();
    } finally {
      try {
        cursorProvider.close();
      } catch (Exception e) {
        LOGGER.error("Error closing cursor provider", e);
      }

      try {
        cursorProvider.releaseResources();
      } catch (Exception e) {
        LOGGER.error("Error releasing cursor provider resources", e);
      }
    }
  }
}
