/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.metadata.handler;

import static org.mule.metadata.api.utils.MetadataTypeUtils.isNotNull;

import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TupleTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;

import java.util.List;

import org.everit.json.schema.ArraySchema;
import org.everit.json.schema.Schema;

public class ArrayHandler implements SchemaHandler {

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof ArraySchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, RestHandlerManager restHandlerManager,
                               ParsingContext parsingContext) {
    ArraySchema arraySchema = (ArraySchema) schema;

    if (arraySchema.getItemSchemas() != null && !arraySchema.getItemSchemas().isEmpty()) {
      final List<Schema> itemSchemas = arraySchema.getItemSchemas();
      final TupleTypeBuilder tupleType = root.tupleType();
      final String description = arraySchema.getDescription();
      if (isNotEmpty(description)) {
        tupleType.description(description);
      }

      final String title = arraySchema.getTitle();
      if (isNotEmpty(title)) {
        tupleType.label(title);
      }
      for (Schema itemSchema : itemSchemas) {
        tupleType.of(restHandlerManager.handle(itemSchema, parsingContext));
      }
      return tupleType;
    } else {

      final ArrayTypeBuilder arrayMetadataBuilder = root.arrayType();
      final String description = arraySchema.getDescription();
      if (isNotEmpty(description)) {
        arrayMetadataBuilder.description(description);
      }

      final String title = arraySchema.getTitle();
      if (isNotEmpty(title)) {
        arrayMetadataBuilder.label(title);
      }

      if (arraySchema.needsUniqueItems()) {
        arrayMetadataBuilder.uniques();
      }

      Integer min = null;
      Integer max = null;

      if (arraySchema.getMinItems() != null && arraySchema.getMinItems() > 0) {
        min = arraySchema.getMinItems();
      }

      if (arraySchema.getMaxItems() != null && arraySchema.getMaxItems() > 0) {
        max = arraySchema.getMaxItems();
      }

      if (isNotNull(min) || isNotNull(max)) {
        arrayMetadataBuilder.boundary(min, max);
      }

      final Schema arrayType = arraySchema.getAllItemSchema();
      if (arrayType != null) {
        arrayMetadataBuilder.of(restHandlerManager.handle(arrayType, parsingContext));
      } else {
        arrayMetadataBuilder.of().anyType();
      }

      final String id = arraySchema.getId();
      if (isNotEmpty(id)) {
        arrayMetadataBuilder.id(id);
      }

      return arrayMetadataBuilder;
    }
  }

  private boolean isNotEmpty(String description) {
    return description != null && !description.isEmpty();
  }
}
