/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.connection;

import static java.lang.String.format;
import static com.mulesoft.connectivity.rest.commons.api.connection.validation.ConnectionValidator.validateConnectionResponse;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.CONNECTIVITY;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.TIMEOUT;
import static com.mulesoft.connectivity.rest.commons.api.error.RestError.getErrorByCode;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.closeStream;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.containsIgnoreCase;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;
import static org.mule.runtime.api.metadata.MediaType.parse;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;

import com.mulesoft.connectivity.rest.commons.api.connection.validation.ConnectionValidationSettings;
import com.mulesoft.connectivity.rest.commons.api.error.RequestException;
import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.api.operation.EntityRequestParameters;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.internal.util.ConnectionValidationUtils;

import java.io.InputStream;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.function.BiConsumer;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link RestConnection}
 *
 * @since 1.0
 */
public class DefaultRestConnection implements RestConnection {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultRestConnection.class);
  private static final String REMOTELY_CLOSED = "Remotely closed";

  private final String baseUri;
  private final String configName;
  private final HttpClient httpClient;
  private final HttpAuthentication authentication;
  private final MultiMap<String, String> defaultQueryParams;
  private final MultiMap<String, String> defaultHeaders;

  /**
   * Creates a new instance
   *
   * @param baseUri            the service base uri
   * @param configName         the name of the config that owns this connection
   * @param httpClient         the client to use
   * @param authentication     the authentication mechanism to use, or {@code null}
   * @param defaultQueryParams query params to be automatically added to all requests done through this connection
   * @param defaultHeaders     headers to be automatically added to all requests done through this connection
   */
  public DefaultRestConnection(String baseUri,
                               String configName,
                               HttpClient httpClient,
                               HttpAuthentication authentication,
                               MultiMap<String, String> defaultQueryParams,
                               MultiMap<String, String> defaultHeaders) {
    this.baseUri = baseUri;
    this.configName = configName;
    this.httpClient = httpClient;
    this.authentication = authentication;
    this.defaultQueryParams = nullSafe(defaultQueryParams);
    this.defaultHeaders = nullSafe(defaultHeaders);
  }

  private void merge(MultiMap<String, String> defaultValues,
                     Predicate<String> appendPredicate,
                     BiConsumer<String, List<String>> appender) {

    defaultValues.keySet().forEach(k -> {
      if (appendPredicate.test(k)) {
        appender.accept(k, defaultValues.getAll(k));
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  public CompletableFuture<Result<String, HttpResponseAttributes>> bodylessRequest(RestRequestBuilder requestBuilder,
                                                                                   int responseTimeoutMillis,
                                                                                   MediaType defaultResponseMediaType,
                                                                                   StreamingHelper streamingHelper) {

    CompletableFuture<Result<String, HttpResponseAttributes>> future = new CompletableFuture<>();

    request(requestBuilder, responseTimeoutMillis, defaultResponseMediaType, streamingHelper).whenComplete((r, e) -> {
      if (e != null) {
        future.completeExceptionally(e);
      } else {
        try {
          Result.Builder<String, HttpResponseAttributes> voidResult = Result.<String, HttpResponseAttributes>builder()
              .output("")
              .length(0);

          r.getAttributes().ifPresent(voidResult::attributes);
          r.getMediaType().ifPresent(voidResult::mediaType);
          r.getAttributesMediaType().ifPresent(voidResult::attributesMediaType);

          future.complete(voidResult.build());

        } finally {
          closeStream(r.getOutput());
        }
      }
    });

    return future;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CompletableFuture<Result<InputStream, HttpResponseAttributes>> request(RestRequestBuilder requestBuilder,
                                                                                int responseTimeoutMillis,
                                                                                MediaType defaultResponseMediaType,
                                                                                StreamingHelper streamingHelper) {

    CompletableFuture<Result<InputStream, HttpResponseAttributes>> future = new CompletableFuture<>();

    HttpRequest request = getHttpRequest(requestBuilder);
    try {
      httpClient.sendAsync(request, HttpRequestOptions.builder()
          .responseTimeout(responseTimeoutMillis)
          .followsRedirect(true)
          .authentication(authentication)
          .build()).whenComplete((response, t) -> {
            try {
              if (t != null) {
                handleRequestException(t, request, future);
              } else {
                handleResponse(response, defaultResponseMediaType, future, streamingHelper);
              }
            } catch (MuleRuntimeException e) {
              future.completeExceptionally(e);
            } catch (Throwable e) {
              future.completeExceptionally(new MuleRuntimeException(
                                                                    createStaticMessage("Unhandled exception on completing send async."),
                                                                    e));
            }
          });
    } catch (Throwable t) {
      handleRequestException(t, request, future);
    }

    return future;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectionValidationResult validate(ConnectionValidationSettings settings, int responseTimeoutMillis) {

    RestRequestBuilder requestBuilder = new RestRequestBuilder(getBaseUri(),
                                                               settings.getTestConnectionPath(),
                                                               settings.getHttpMethod(),
                                                               new EntityRequestParameters());
    HttpResponse response;

    try {
      response = httpClient.send(getHttpRequest(requestBuilder),
                                 HttpRequestOptions.builder()
                                     .responseTimeout(responseTimeoutMillis)
                                     .followsRedirect(true)
                                     .authentication(authentication)
                                     .build());
    } catch (TimeoutException e) {
      return ConnectionValidationUtils.connectionValidationResult(new ModuleException(e.getMessage(), TIMEOUT, e));
    } catch (Throwable t) {
      return ConnectionValidationUtils.connectionValidationResult(new ModuleException(t.getMessage(), CONNECTIVITY, t));
    }

    return validateConnectionResponse(response, settings);
  }

  private HttpRequest getHttpRequest(RestRequestBuilder requestBuilder) {
    final MultiMap<String, String> headers = requestBuilder.getHeaders();
    final MultiMap<String, String> queryParams = requestBuilder.getQueryParams();

    merge(defaultHeaders, k -> !headers.containsKey(k), requestBuilder::addHeaders);
    merge(defaultQueryParams, k -> !queryParams.containsKey(k), requestBuilder::addQueryParams);

    return buildRequest(requestBuilder);
  }

  private void handleResponse(HttpResponse response,
                              MediaType defaultResponseMediaType,
                              CompletableFuture<Result<InputStream, HttpResponseAttributes>> future,
                              StreamingHelper streamingHelper) {

    RestError error = getErrorByCode(response.getStatusCode()).orElse(null);
    if (error != null) {
      handleResponseError(response, defaultResponseMediaType, future, streamingHelper, error);
    } else {
      future.complete(toResult(response, false, defaultResponseMediaType, streamingHelper));
    }
  }

  protected void handleResponseError(HttpResponse response,
                                     MediaType defaultResponseMediaType,
                                     CompletableFuture<Result<InputStream, HttpResponseAttributes>> future,
                                     StreamingHelper streamingHelper,
                                     RestError error) {
    future
        .completeExceptionally(new RequestException(error, toResult(response, true, defaultResponseMediaType, streamingHelper)));
  }

  private void handleRequestException(Throwable t,
                                      HttpRequest request,
                                      CompletableFuture<Result<InputStream, HttpResponseAttributes>> future) {

    checkIfRemotelyClosed(t, request);
    RestError error = t instanceof TimeoutException ? TIMEOUT : CONNECTIVITY;
    future.completeExceptionally(new ModuleException(t.getMessage(), error, t));
  }

  private <T> Result<T, HttpResponseAttributes> toResult(HttpResponse response,
                                                         boolean isError,
                                                         MediaType defaultResponseMediaType,
                                                         StreamingHelper streamingHelper) {
    Result.Builder<T, HttpResponseAttributes> builder = Result.builder();

    HttpEntity entity = response.getEntity();
    Object content = entity.getContent();
    if (isError) {
      content = streamingHelper != null ? streamingHelper.resolveCursorProvider(content) : content;
    }

    builder.output((T) content);
    entity.getBytesLength().ifPresent(builder::length);

    MediaType contentType = defaultResponseMediaType;
    String responseContentType = response.getHeaders().get(CONTENT_TYPE);
    if (responseContentType != null) {
      try {
        contentType = parse(responseContentType);
      } catch (Exception e) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(format("Response Content-Type '%s' could not be parsed to a valid Media Type. Will ignore",
                              responseContentType),
                       e);
        }
      }
    }

    builder.mediaType(contentType);
    builder.attributes(toAttributes(response)).attributesMediaType(APPLICATION_JAVA);

    return builder.build();
  }

  protected HttpRequest buildRequest(RestRequestBuilder requestBuilder) {
    return requestBuilder.build();
  }

  protected HttpResponseAttributes toAttributes(HttpResponse response) {
    return new HttpResponseAttributes(response.getStatusCode(), response.getReasonPhrase(), response.getHeaders());
  }

  /**
   * Template method invoked by the {@link #stop()} to perform custom actions before the actual stopping
   * takes place. This method should not fail, but if it does, it won't stop the actual stopping process
   */
  protected void beforeStop() {
    // default implementation is empty
  }

  /**
   * Template method invoked by the {@link #stop()} to perform custom actions after the actual stopping
   * takes place. This method should not fail, but if it does, it won't stop the actual stopping process
   */
  protected void afterStop() {
    // default implementation is empty
  }

  private void checkIfRemotelyClosed(Throwable exception, HttpRequest request) {
    if ("https".equals(request.getUri().getScheme()) && containsIgnoreCase(exception.getMessage(), REMOTELY_CLOSED)) {
      LOGGER.error("Remote host closed connection. Possible SSL/TLS handshake issue. Check protocols, cipher suites and "
          + "certificate set up. Use -Djavax.net.debug=ssl for further debugging.");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getBaseUri() {
    return baseUri;
  }

  /**
   * Stops this connection by stopping the underlying {@link HttpClient} and freeing other resources.
   * <p>
   * It will perform that task around invokations of {@link #beforeStop()} and {@link #afterStop()}, which
   * allow implementors to customize the stopping process. Notice that those methods should not fail, but if
   * they do, they won't stop resources from being freed.
   */
  @Override
  public final void stop() {
    try {
      beforeStop();
    } catch (Throwable t) {
      LOGGER.warn(format("Exception found before stopping config '%s'", configName), t);
    }

    try {
      httpClient.stop();
    } catch (Throwable t) {
      LOGGER.warn(format("Exception found while stopping http client for config '%s'", configName), t);
    }

    try {
      afterStop();
    } catch (Throwable t) {
      LOGGER.warn(format("Exception found after stopping config '%s'", configName), t);
    }
  }

  private MultiMap<String, String> nullSafe(MultiMap<String, String> multiMap) {
    return multiMap != null ? multiMap : new MultiMap<>();
  }
}
